/*
	Sunrise Calculator - A Sunrise and Sunset Calculator
	Copyright © 2006-2015 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

	Sunrise Calculator - browser version 1.0
	30 April, 2015
	Copyright © 2006-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global paramOne, paramTwo, paramThree, paramFour, paramFive, paramSix,
	headingOne, headingTwo, headingThree, headingFour, clearData, Menu,
	parseLatOrLong, calculate, moonRiseSet, menuMap, setUp,
	maxWidth
*/


// map custom names onto model names
var latitude  = paramOne;
var longitude = paramTwo;
var gregorian = paramThree;
var sunrise   = paramFour;
var altitude  = paramFive;
var timezone  = paramSix;
/*
var latitudeHeading  = headingOne;
var longitudeHeading = headingTwo;
var gregorianHeading = headingThree;
var sunriseHeading   = headingFour;
*/

var calcTypeMenu = Menu[0];
var unitsMenu    = Menu[1];
var timeZoneMenu = Menu[2];

var calcType = 0;					// Official
var units    = 'metres';
var timeZone = 'Universal Time';	// UTC

/*
	['Official', 'Civil', 'Nautical', 'Astronomical'],
	['feet', 'metres'],
	['My Local Time', 'Universal Time', 'As given below']
*/

var zenithPref     = 90.8333;
var timeZonePref   = 'Universal Time';
var timeZoneOffset = 0;

var latitudePref  =  '53 39 28 N';
var longitudePref = '001 56 48 W';

var altitudePref = 0;
var altitudeUnitsPref = 'metres';

var altitudeMetresMax = 15240;	// 15240 metres
var altitudeFeetMax   = 50000;	// 50000 feet

function Position(latitude, longitude) {
	this.latitude  = latitude;
	this.longitude = longitude;
}

function CalDate(year, month, day) {
	this.year  = year;
	this.month = month;
	this.day   = day;
}

function today() {
	var d  = new Date();
	
	return new CalDate(d.getFullYear(), d.getMonth() + 1, d.getDate());
}

function leapYear(year) {
	if ((year %   4) !== 0) { return false; }
	if ((year % 400) === 0) { return true; }
	if ((year % 100) === 0) { return false; }
	return true;
}

function getDaysIn(month, year) {	// month 1..12
	var daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
	
	if (month !== 2) { return daysInMonth[month - 1]; }
	if (leapYear(year)) { return 29; }
	return 28;
}

function parseDate(date) {
	var invalidDate = new CalDate(0, 0, 0),
		item = date.split('/', 3),
		year,
		month,
		day;
		
	if (item.length !== 3) { return invalidDate; }
	
	year   = item[0];
	month  = item[1];
	day    = item[2];
	
	year  = parseInt(year,  10);

	if (isNaN(year)) { return invalidDate; }
	month = parseInt(month, 10);
	if (isNaN(month)) { return invalidDate; }
	day   = parseInt(day,   10);
	if (isNaN(day)) { return invalidDate; }

	if ((year  < 1970) || (year  > 9999)) { return invalidDate; }
	if ((month <    1) || (month >   12)) { return invalidDate; }
	if ((day   <    1) || (day   >   31)) { return invalidDate; }
	
	if (day > getDaysIn(month, year)) { return invalidDate; }
	
	return new CalDate(year, month, day);
}

function get_ang(data, dmax, NE, SW) {
	switch (dmax) {
	case 90:
		return parseLatOrLong(data, true, false, " ");
	case 180:
		return parseLatOrLong(data, false, false, " ");
	default:
		return NaN;
	}
}

function setSunriseTooltip(localOffset, altitude, units) {
	var tooltip = 'Sunrise/Sunset times expressed as times in zone GMT ',
		offset = Math.round(60 * localOffset),
		sign = "+",
		mins,
		hours;

	if (offset < 0) { sign = "-"; offset = -offset; }
	mins  = offset % 60;
	hours = Math.round((offset - mins) / 60);
	hours = String(hours);
	mins  = String(mins);
	if (hours.length === 1) { hours = "0" + hours; }
	if (mins.length  === 1) { mins  = "0" + mins;  }
	
	sunrise.title = tooltip + sign + hours + mins + '. Altitude set at ' + altitude + ' ' + units + '.' +
		'\n\nMoonrise/Moonset times are displayed on the right.';
}

function sunRiseSet(date, position, altitude, units) {
	var latitude  = get_ang(position.latitude,   90, "N", "S"),
		longitude = get_ang(position.longitude, 180, "E", "W"),
		year,
		month,
		day,
		theDate,
		localOffset,
		tzOffset,
		theta,
		sunrise,
		sunset,
		zenith;
	
	if (isNaN(latitude)) { return 'Invalid Latitude';  }
	if (isNaN(longitude)) { return 'Invalid Longitude'; }

	year  = date.year;
	
	if (year === 0) { return 'Invalid Date'; }
	
	month = date.month;
	day   = date.day;
	
	theDate   = new Date(year, month - 1, day, 12, 0, 0);
	
	localOffset = timeZoneOffset;	// in hours

	if (timeZoneOffset === 24) {				// use local time
		tzOffset = theDate.getTimezoneOffset();	// in minutes
		localOffset = -tzOffset / 60;
		timezone.value = localOffset;
	}
	
	setSunriseTooltip(localOffset, altitude, units);
	
	theta = 0.0321 * Math.sqrt(altitudePref);	// converts altitude in metres to zenith delta

	zenith  = zenithPref;	// type of sunrise/sunset
	zenith += theta;		// add altitude adjustment
	
	sunrise = calculate(true,  year, month, day, latitude, longitude, zenith, localOffset);
	sunset  = calculate(false, year, month, day, latitude, longitude, zenith, localOffset);

	return sunrise + '/' + sunset + '          ' + moonRiseSet(latitude, longitude, theDate, -localOffset);
}

function update() {		// re-calculate sunrise and sunset
	var P = new Position(latitude.value, longitude.value),
  		D = parseDate(gregorian.value);
  		
 	sunrise.value = sunRiseSet(D, P, altitudePref, altitudeUnitsPref);
}

function setToday() {
	var d = today(),
		year = String(d.year),
		month = String(d.month),
		day = String(d.day);
	
	if (month.length === 1) { month = '0' + month; }
	if (day.length   === 1) { day   = '0' + day;   }

	gregorian.value = year + '/' + month + '/' + day;
	update();
}

function updatePrefs(starting) {
	var K, L;

	zenithPref        = menuMap[calcType];
	altitudeUnitsPref = units;
	timeZonePref      = timeZone;

	if (timeZonePref === 'Universal Time') {
		timeZoneOffset = 0;
		timezone.value = "0";
	} else if (timeZonePref === 'My Local Time') {
		timeZoneOffset = 24;	// marker for local time
		timezone.value = "";
	} else {
		if (timezone.value === "") { timezone.value = "0"; }
		timeZoneOffset = parseFloat(timezone.value);
		if (isNaN(timeZoneOffset)) {
			timeZoneOffset = 48;
		} else {
			timeZoneOffset = Math.round(4 * timeZoneOffset) / 4;
			if ((timeZoneOffset > 14) || (timeZoneOffset < -12)) { timeZoneOffset = 48; }
		}
		if (timeZoneOffset === 48) {
			alert('Invalid Time Zone Offset has been ignored.');
			timeZoneOffset = 0;
			timezone.value = "";
		} else { timezone.value = timeZoneOffset.toString(); }
	}

	K = get_ang(latitudePref,   90, "N", "S");
	L = get_ang(longitudePref, 180, "E", "W");
	
	if (starting) {
		if (!isNaN(K) && !isNaN(L)) {	// we have a valid position, so use it
  			latitude.value  = latitudePref;
  			longitude.value = longitudePref;
  		} else { alert('Invalid Starting Location has been ignored.'); }
  	}
  			
  	if (altitude.value === "") { altitude.value = "0"; }
  	altitudePref = parseFloat(altitude.value);
 	if (isNaN(altitudePref)) { altitudePref = 0; }
 	if (altitudePref < 0) { altitudePref = 0; }
 	if ((altitudeUnitsPref === "metres") && (altitudePref > altitudeMetresMax)) {
 		altitudePref = altitudeMetresMax;
 	} else if ((altitudeUnitsPref === "feet") && (altitudePref > altitudeFeetMax)) {
 		altitudePref = altitudeFeetMax;
 	}
 	altitude.value = String(altitudePref);
  	if (altitudeUnitsPref === "feet") { altitudePref *= 0.3048; }	// convert feet to metres

	update();
//	setUp(maxWidth);
}

calcTypeMenu.onchange  = function () {
	calcType = this.selectedIndex;
	updatePrefs();
};

unitsMenu.onchange  = function () {
	units = this.value;
	updatePrefs();
};

timeZoneMenu.onchange  = function () {
	timeZone = this.value;
	updatePrefs();
};


latitude.onkeypress = function () {
	var x = event.which || event.keyCode;
	
	if (x === 13) { updatePrefs(); } else { sunrise.value = ""; }
};

longitude.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { updatePrefs(); } else { sunrise.value = ""; }
};

gregorian.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { updatePrefs(); } else { sunrise.value = ""; }
};

altitude.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { updatePrefs(); } else { sunrise.value = ""; }
};

timezone.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { updatePrefs(); } else { sunrise.value = ""; }
};

setToday();
updatePrefs(true);
