/*
    Resistors - A special purpose calculator to decode the colored bands on
    resistors (and small capacitors).
    Copyright © 2004-2020 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License as published by the Free Software
    Foundation; either version 2 of the License, or (at your option) any later
    version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE.  See the GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along with
    this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
    St, Fifth Floor, Boston, MA  02110-1301  USA

    Resistors - browser version 3.0
    10 May, 2020
    Copyright 2004-2020 Harry Whitfield
    mailto:g6auc@arrl.net
*/

/*jslint browser, devel, this */

/*property
    checked, forEach, index, length, onchange, onclick, onload, onmousedown,
    onmouseup, opacity, open, pow, push, round, selectedIndex, src, style,
    title, toPrecision, toString, value, visibility
*/

import {newCheckbox, newImage, newInput, newSelector, addToMenu} from "./webWidget.js";
import {getPreference, setPreference} from "./preferences.js";
import {initSound, playSound} from "./sounds.js";
import {buzzerBuffer, keyPressBuffer, cReturnBuffer} from "./sounds.js";

///////////////////////////// Start of the resistors skeleton ////////////////////////////

// newImage(hOffset, vOffset, width, height, src, zOrder, opacity)
// newInput(hOffset, vOffset, width, height, value, zOrder, style, id)
// newSelector(hOffset, vOffset, width, height, src, zOrder, opacity)
// newCheckbox(hOffset, vOffset, width, height, value, zOrder, style, id)

var componentPref = getPreference("componentPref");
var bandsPref = getPreference("bandsPref");
var internationalPref = getPreference("internationalPref");
var componentImagePref = getPreference("componentImagePref");

var mBand = Number(bandsPref);

var band = 1;   // takes values 1..4 and represents the state of the state machine
var value = 0;
var key = [];
var bands = [];

/*
var colors = [
	"#000000", "#812900", "#FF0000", "#FF6E00", "#FFFB0A", "#37FF10", "#140CFF", "#E400FF", "#8D8D8D", "#FFFFFF", "#FFC400", "#CECECE"
];
*/

var tooltips = [
	"black x 1", "brown x 10  1% (F)  100ppm", "red x 100  2% (G)  50ppm", "orange x 1000  15ppm", "yellow x 10,000  25ppm", "green x 100,000  0.5% (D)",
    "blue x 1,000,000  0.25% (C)", "violet x 10,000,000  0.1% (B)", "gray x 100,000,000  0.05% (A)", "white x 1,000,000,000", "gold", "silver"
];

var style = "font-size:18px;font-family:Times;font-style:italic;color:black;background-color:transparent;border:none;";

//			   newInput(hOffset, vOffset, width, height, value, zOrder, style, id)
var resValue = newInput(23, 236, 108, 20, "", 10, style);

var swbase = "Resources/Swatches/";
var swatch1 = newImage(215, 244, 8, 16, swbase + "swatch4.png", 3, 0);
var swatch2 = newImage(227, 244, 8, 16, swbase + "swatch7.png", 3, 0);
var swatch3 = newImage(239, 244, 8, 16, swbase + "swatch6.png", 3, 0);
var swatch4 = newImage(251, 244, 8, 16, swbase + "swatch9.png", 3, 0);

function buildSwatches(color1, color2, color3, color4) {
    "use strict";

    if ((componentImagePref === "1") && (color1 !== undefined)) {
        swatch1.src = swbase + "swatch" + String(color1) + ".png";
        swatch1.style.opacity = 1.0;
    } else {
        swatch1.style.opacity = 0.0;
    }
    if ((componentImagePref === "1") && (color2 !== undefined)) {
        swatch2.src = swbase + "swatch" + String(color2) + ".png";
        swatch2.style.opacity = 1.0;
    } else {
        swatch2.style.opacity = 0.0;
    }
    if ((componentImagePref === "1") && (color3 !== undefined)) {
        swatch3.src = swbase + "swatch" + String(color3) + ".png";
        swatch3.style.opacity = 1.0;
    } else {
        swatch3.style.opacity = 0.0;
    }
    if ((componentImagePref === "1") && (color4 !== undefined)) {
        swatch4.src = swbase + "swatch" + String(color4) + ".png";
        swatch4.style.opacity = 1.0;
    } else {
        swatch4.style.opacity = 0.0;
    }
}

function show(value, multiplier, unit) {    // multiplier -2..1
    "use strict";
    var ls;
    var ms;

    if (value > 99) {   // occurs when mBand === 4
        switch (multiplier) {
        case 1:
            return value + unit;
        case 0:
            ls = value % 10;
            ms = Math.round((value - ls) / 10);
            return ms + unit + ls;
        case -1:
            ls = value % 100;
            ms = Math.round((value - ls) / 100);
            ls = String(ls);
            if (ls.length === 1) {
                ls = "0" + ls;
            }
            return ms + unit + ls;
        default:
            playSound(buzzerBuffer);
            return "";
        }
    }
    // occurs when mBand === 3
    switch (multiplier) {
    case 1:
        return 10 * value + unit;
    case 0:
        return value + unit;
    case -1:
        ls = value % 10;
        ms = Math.round((value - ls) / 10);
        return ms + unit + ls;
    case -2:
        return "0" + unit + value;
    default:
        playSound(buzzerBuffer);
        return "";
    }
}

function internationalFormat(value, multiplier) {   // multiplier -2..9
    "use strict";
    var prefix = ["p", "n", "µ", "m", "R", "k", "M", "G"];
    var prBase = 0;

    if (componentPref === "Resistor") {
        prBase = 4;
    }

    if (value > 99) {
        multiplier += 1;
    }    // occurs when mBand === 4

    if (multiplier >= 8) {
        return show(value, multiplier - 9, prefix[prBase + 3]);
    }
    if (multiplier >= 5) {
        return show(value, multiplier - 6, prefix[prBase + 2]);
    }
    if (multiplier >= 2) {
        return show(value, multiplier - 3, prefix[prBase + 1]);
    }
    return show(value, multiplier, prefix[prBase]);
}

function display(value, multiplier) {   // multiplier -2..9
    "use strict";
    var strValue;
    var temp = value;
    var color4;

    if (internationalPref === "0") {    // use integer format
        temp = temp * Math.pow(10, multiplier);
        if (multiplier < 0) {
            strValue = temp.toPrecision(mBand - 1);
        } else {
            strValue = temp.toString();
        }
        resValue.value = strValue;
        if (multiplier > 7) {
            strValue = internationalFormat(value, multiplier);
        }
    } else {    // use international format
        strValue = internationalFormat(value, multiplier);
        resValue.value = strValue;
    }
    color4 = undefined;
    if (mBand === 4) {
        color4 = bands[4];
    }
    buildSwatches(bands[1], bands[2], bands[3], color4);
}

function process() {// handles keys 0 to 9
    "use strict";
    var s = this.index;

    this.style.opacity = 0.5;

    switch (band) {
    case 1:
        if (s !== 0) {
            buildSwatches();
            bands[1] = Number(s);
            resValue.value = s;
            value = Number(s);
            band = 2;
            playSound(keyPressBuffer);
            return;
        }
        if (componentPref === "Resistor") {
            buildSwatches(0);
            resValue.value = (
            	internationalPref === "0"
                ? "0"
                : "0R"
            );
            playSound(keyPressBuffer);
            return;
        }
        buildSwatches();
        resValue.value = "";
        playSound(buzzerBuffer);
        return;
    case 2:
        bands[2] = Number(s);
        resValue.value += s;
        value = 10 * value + Number(s);
        band = 3;
        playSound(keyPressBuffer);
        return;
    case 3:
        bands[3] = Number(s);
        if (mBand === 3) {
            display(value, Number(s));
            band = 1;
            playSound(keyPressBuffer);
            return;
        }
        resValue.value += s;
        value = 10 * value + Number(s);
        band = 4;
        playSound(keyPressBuffer);
        return;
    case 4:
        bands[4] = Number(s);
        display(value, Number(s));
        band = 1;
        playSound(keyPressBuffer);
        return;
    default:
        playSound(buzzerBuffer);
        return; // invalid state
    }
}

function processStarHash() { // handles the * and # keys
    "use strict";
    var multiplier = this.index;

    this.style.opacity = 0.5;
    if (band === mBand) {
        bands[band] = 9 - multiplier;
        display(value, multiplier);
        band = 1;
        playSound(keyPressBuffer);
    } else {
       	playSound(buzzerBuffer);
    }
}

function processClear() { // handles the CLEAR key
    "use strict";
    this.style.opacity = 0.5;
    resValue.value = "";
    buildSwatches();
    band = 1;
    playSound(cReturnBuffer);
}

function setOpacity255() {
    "use strict";
    this.style.opacity = 1.0;
}

//////////////////////////////////////////////////////////////////////////////////////////

var picbase = "Resources/Pictures/";

var background = newImage(18, 231, 122, 55, picbase + "roller.png", 5, 1.0);
background.title = "";

var dockBg = newImage(190, 237, 100, 30, picbase + "dockbg.png", 2, 1.0);
dockBg.title = "";

var keyStar = newImage(25, 191, 27, 29, picbase + "star.png", 7, 1.0);
keyStar.index = -2;
keyStar.onmouseup = setOpacity255;
keyStar.onmousedown = processStarHash;
keyStar.title = "silver x 0.01  10% (K)";

var keyHash = newImage(107, 191, 27, 29, picbase + "hash.png", 7, 1.0);
keyHash.index = -1;
keyHash.onmouseup = setOpacity255;
keyHash.onmousedown = processStarHash;
keyHash.title = "gold x 0.1  5% (J)";

var clear = newImage(127, 36, 27, 29, picbase + "CLR.png", 7, 1.0);
clear.onmouseup = setOpacity255;
clear.onmousedown = processClear;
clear.title = "CLEAR";

var pipe = newImage(69, 14, 25, 312, picbase + "pipe.png", 4, 1.0);
pipe.title = "";
var menuarea = newImage(0, 31, 172, 292, picbase + "menuarea.png", 4, 1.0);
menuarea.title = "";
var outerBase = newImage(22, 58, 114, 182, picbase + "outer-base.png", 4, 1.0);
outerBase.title = "";
var innerBase = newImage(20, 57, 121, 191, picbase + "inner-base.png", 4, 1.0);
innerBase.title = "";
var bars = newImage(0, 93, 149, 155, picbase + "bars.png", 4, 1.0);
bars.title = "";
var base = newImage(3, 231, 154, 68, picbase + "base.png", 4, 1.0);
base.title = "";
var slider = newImage(60, 256, 49, 36, picbase + "slider.png", 7, 1.0);
slider.title = "";
var rightBar = newImage(135, 52, 15, 68, picbase + "rightbar.png", 5, 1.0);
rightBar.title = "";
var roundButton1 = newImage(17, 183, 24, 31, picbase + "roundbutton1.png", 5, 1.0);
roundButton1.title = "";
var roundButton2 = newImage(143, 103, 24, 31, picbase + "roundbutton2.png", 5, 1.0);
roundButton2.title = "Click to show/hide the option menus.";
var leftButton1 = newImage(0, 219, 30, 33, picbase + "leftbutton1.png", 5, 1.0);
leftButton1.title = "";
var leftButton2 = newImage(0, 181, 30, 33, picbase + "leftbutton2.png", 5, 1.0);
leftButton2.title = "";
var leftButton3 = newImage(0, 141, 30, 33, picbase + "leftbutton3.png", 5, 1.0);
leftButton3.title = "";
var slot = newImage(139, 141, 22, 84, picbase + "slot.png", 5, 1.0);
slot.title = "";
var knob = newImage(139, 198, 15, 11, picbase + "knob.png", 5, 1.0);
knob.title = roundButton2.title;

function makeKeys() { //creates the images for keys 0 to 9
    "use strict";
	var coords = [[66, 191], [26, 71], [66, 71], [106, 71], [26, 111], [66, 111], [106, 111], [26, 151], [66, 151], [106, 151]];

    function doThis(coord, i) {
        var img = newImage(coord[0], coord[1], 28, 28, picbase + String(i) + ".png", 7, 1.0);
        img.index = i;
        img.onmouseup = setOpacity255;
        img.onmousedown = process;
        img.title = tooltips[i];
        key.push(img);
    }
    coords.forEach(doThis);
}

makeKeys();
resValue.value = "";

//////////////////////////////////////////////////////////////////////////////////////////

var helpButton = newImage(190, 36, 30, 22, picbase + "help.png", 2);   // customized
helpButton.title = "Displays information about this program.";
helpButton.onmousedown = function () {
    "use strict";
    this.style.opacity = "0.5";
};
helpButton.onmouseup = function () {
    "use strict";
    this.style.opacity = "1.0";
    window.open("Help.html");
};

var resistor = newImage(200, 237, 75, 30, picbase + "resistor.png", 2, 1.0);
resistor.title = "";

//////////////////////////////////// Menus Customized ////////////////////////////////////

var MenuArray = [
    ["Resistor", "Capacitor"],
    ["3", "4"],
    ["Numerical", "International"]
];

var MenuTitleArray = [
    "Choose the type of component to be decoded.",
    "Choose the number of bands used to encode the value of the component.",
    "Choose how component values are to be displayed."
];

//var MenuSelectedIndex = [0, 0, 0];
var Menu = [];

Menu[0] = newSelector(190, 68 + 4, 110, 20, "", 2);
Menu[1] = newSelector(190, 68 + 46, 110, 20, "", 2);
Menu[2] = newSelector(190, 68 + 88, 110, 20, "", 2);


(function () {
    "use strict";
    MenuArray.forEach(function (ele, i) {
        addToMenu(Menu[i], ele);
    //	Menu[i].selectedIndex = MenuSelectedIndex[i];
        Menu[i].title = MenuTitleArray[i];
    });

    Menu[0].value = componentPref;
    Menu[1].value = bandsPref;
    Menu[2].value = MenuArray[2][Number(internationalPref)];
}());

//////////////////////////////////////////////////////////////////////////////////////////

function hideComponentImage() {
	dockBg.style.visibility = "hidden";
	resistor.style.visibility = "hidden";
}

function showComponentImage() {
	dockBg.style.visibility = "visible";
	resistor.style.visibility = "visible";
}

//					newCheckbox(hOffset, vOffset, width, height, value, zOrder, style, id)
var showCompImage = newCheckbox(190, 68 + 130, 20, 20, componentImagePref, 20, "");
showCompImage.title = "Check this box if an image of the component is to be displayed.";

if (componentImagePref === "1") {
	showComponentImage();
} else {
	hideComponentImage();
}

//////////////////////////////////////////////////////////////////////////////////////////

var componentPrefMenu = Menu[0];
var bandsPrefMenu = Menu[1];
var internationalPrefMenu = Menu[2];

var timeoutID1;

function hideMenus() {
	Menu.forEach(function (ele) {
		ele.style.visibility = "hidden";
	});
	helpButton.style.visibility = "hidden";
	showCompImage.style.visibility = "hidden";
}

function showMenus() {
	Menu.forEach(function (ele) {
		ele.style.visibility = "visible";
	});
	helpButton.style.visibility = "visible";
	showCompImage.style.visibility = "visible";
}

knob.onclick = function () {
	if (Menu[0].style.visibility === "visible") {
		clearTimeout(timeoutID1);
		hideMenus();
	} else {
		showMenus();
		timeoutID1 = setTimeout(hideMenus, 20000);
	}
};

roundButton2.onclick = knob.onclick;

componentPrefMenu.onchange = function () {
    "use strict";
    componentPref = this.value;
    setPreference("componentPref", componentPref);
    band = 1;
    resValue.value = "";
    buildSwatches();
};

bandsPrefMenu.onchange = function () {
    "use strict";
    bandsPref = this.value;
    setPreference("bandsPref", bandsPref);
    mBand = Number(bandsPref);
    band = 1;
    resValue.value = "";
    buildSwatches();
};

internationalPrefMenu.onchange = function () {
    "use strict";
    internationalPref = String(this.selectedIndex);
    setPreference("internationalPref", internationalPref);
    band = 1;
    resValue.value = "";
    buildSwatches();
};

showCompImage.onchange = function () {
	componentImagePref = (
		showCompImage.checked
		? "1"
		: "0"
	);
	setPreference("componentImagePref", componentImagePref);
	if (componentImagePref === "1") {
		showComponentImage();
	} else {
		hideComponentImage();
	}
	buildSwatches();
};

window.onload = function () {
	hideMenus();
	initSound();
};
