/*	Based on calculator.js, by Daniel Dickison (danieldickison@gmail.com),
	which handled input and math for the Kalculator widget.
	© 2004 Daniel Dickison

	This version is part of the Roman Kalculator widget by Harry Whitfield (g6auc@arrl.net)
	© 2016 Daniel Dickison and Harry Whitfield
*/

/*property
	E, LOG10E, LOG2E, PI, acos, asin, atan, ceil, cos, exp, floor, indexOf,
	length, log, pow, sin, sqrt, substring, tan, toPrecision, toString
*/

/*jslint bitwise */

"use strict";

var displayNumber;
var displayOp;
var displayMemory;
var setOpSet;
var logToFile;
var logDisplay;
var print;
var beep;

//////////////////////////////////////////////////////////////////////////////////////////

var numString = "";
var n = 0;
var x = NaN;
var prevN = NaN;
var op = "";
var prevOp = "";
var memory = 0;
var degreeMode;

var reporting = false;

function report(fn) {
	if (!reporting) {
		return;
	}
	print("--- " + fn + " ---");
	print("numString: " + numString);
	print("op:		   " + op);
	print("prevOp:	   " + prevOp);
	print("n:		   " + n);
	print("prevN:	   " + prevN);
	print("x:		   " + x);
}

function rOfA(a) {
	return (degreeMode)
		? Math.PI * a / 180
		: a;
}

function aOfR(a) {
	return (degreeMode)
		? 180 * a / Math.PI
		: a;
}

function compute1(op, a, b) {
	switch (op) {
	case "⌈ ⌉":
		return Math.ceil(a);
	case "⌊ ⌋":
		return Math.floor(a);
	case "x⁻¹":
		return 1 / a;
	case "√x":
		return Math.sqrt(a);
	case "x²":
		return a * a;
	case "±":
		return -a;
	case "sinh":
		return 0.5 * (Math.exp(a) - Math.exp(-a));
	case "cosh":
		return 0.5 * (Math.exp(a) + Math.exp(-a));
	case "tanh":
		return (Math.exp(a) - Math.exp(-a)) / (Math.exp(a) + Math.exp(-a));
	case "asinh":
		return Math.log(a + Math.sqrt(a * a + 1));
	case "acosh":
		return Math.log(a + Math.sqrt(a * a - 1));
	case "atanh":
		return 0.5 * Math.log((1 + a) / (1 - a));
	case "exp":
		return Math.exp(a);
	case "10ⁿ":
		return Math.pow(10, a);
	case " 2ⁿ":
		return Math.pow(2, a);
	case "sin":
		return Math.sin(rOfA(a));
	case "cos":
		return Math.cos(rOfA(a));
	case "tan":
		return Math.tan(rOfA(a));
	case "log2":
		return Math.LOG2E * Math.log(a);
	case "ln ":
		return Math.log(a);
	case "log":
		return Math.LOG10E * Math.log(a);
	case "asin":
		return aOfR(Math.asin(a));
	case "acos":
		return aOfR(Math.acos(a));
	case "atan":
		return aOfR(Math.atan(a));
	case "^":
		return Math.pow(a, b);
	case "√":
		return Math.pow(a, 1 / b);
	case "%":
		return a % b;
	case "÷":
		return (a - a % b) / b;
	case "/":
		return a / b;
	case "*":
		return a * b;
	case "-":
		return a - b;
	case "+":
		return a + b;
	case "~":
		return ~a;
	case "&":
		return a & b;
	case "|":
		return a | b;
	case "⊕":
		return a ^ b;
	case "<<":
		return a << b;
	case ">>":
		return a >> b;
	case ">>>":
		return a >>> b;
	}
}

function compute(op, a, b) {
	var result = compute1(op, a, b);

	print("computing " + a + " " + op + " " + b + ",  result = " + result);

	logDisplay("compute");
	logToFile("computing " + a + " " + op + " " + b + ",  result = " + result + "\n");
	return result;
}

function isUnary(op) {
	return (op === "x⁻¹" || op === "√x" || op === "x²" || op === "±" ||
			op === "sinh" || op === "cosh" || op === "tanh" || op === "exp" ||
			op === "asinh" || op === "acosh" || op === "atanh" || op === "10ⁿ" ||
			op === "sin" || op === "cos" || op === "tan" || op === "ln " ||
			op === "asin" || op === "acos" || op === "atan" || op === "log" ||
			op === "⌈ ⌉" || op === "⌊ ⌋" || op === " 2ⁿ" || op === "log2" || op === "~");
}

function displayPrevOp() {
	var prevStr;

	function showOp(op, str) {
		if (op === "√") {
			displayOp(str + op);
		} else {
			displayOp(op + " " + str);
		}
	}

	if (isNaN(prevN) || (prevOp === "")) {
		displayOp("");
	} else if (isUnary(prevOp)) {
		displayOp(prevOp);
	} else {
		prevStr = prevN.toString();
		if (prevStr.length > 6) {
			showOp(prevOp, prevN.toPrecision(7));
		} else {
			showOp(prevOp, prevStr);
		}
	}
}

function doConstant(item) {
	if (item === "π") {
		n = Math.PI;
	} else if (item === "e") {
		n = Math.E;
	}
	displayNumber(String(n));
	numString = "";
	logDisplay("doConstant");
}

function doEnter() {
	report("doEnter");

	if (op === "") {
		if ((prevOp === "") || isNaN(prevN)) {
			return;
		}
		op = prevOp;
		x = n;
		n = prevN;
	}
	prevN = n;
	prevOp = op;
	displayPrevOp();
	n = compute(op, x, n);
	x = NaN;
	displayNumber(String(n));
	numString = "";
	op = "";
	setOpSet(false);

	logDisplay("doEnter");
	report("doEnter end");
}

function doOperator(item) {
	report("doOperator");

	if (isNaN(x) && (op === "")) {
		x = n;
		n = 0;
	} else if ((op !== "") && (numString !== "")) {
		x = compute(op, x, n);
		displayNumber(String(x));
	}
	numString = "";
	op = item;
	setOpSet(true);
	displayOp(op);
	if (isUnary(op)) {
		doEnter();
	}

	logDisplay("doOperator");
	report("doOperator end");
}

function doNumber2(num) {
	if ((numString === "") && (op === "")) {
		x = NaN;
	}

	numString = num;
	n = Number(numString);
}

function doNumber(item) {
	report("doNumber");

	if ((numString === "") && (op === "")) {
		x = NaN;
	}

	// Refuse leading zeros except in this one case "0."
	if ((numString === "0") && (item !== ".")) {
		beep();
		return;
	}

	// Refuse repeated periods.
	if ((numString.indexOf(".") !== -1) && (item === ".")) {
		beep();
		return;
	}

	// Refuse repeated Es.
	if ((numString.indexOf("E") !== -1) && (item === "E")) {
		beep();
		return;
	}

	// Refuse - except in exponent
	if ((item === "+") || (item === "-")) {
		if (numString[numString.length - 1] !== "E") {
			doOperator(item);
			return;
		}
	}

	numString += item;
	n = Number(numString);
	displayNumber(numString);

	report("doNumber end");
}

function doMemory(op) {
	var memStr;

	report("doMemory");

	if (op === "mr") {
		n = memory;
		displayNumber(n);
		numString = "";
		return;
	}
	if (op === "mc") {
		memory = 0;
	} else if (op === "m+") {
		memory += n;
	} else if (op === "m-") {
		memory -= n;
	}
	memStr = memory.toString();
	if (memStr.length > 8) {
		displayMemory("M: " + memory.toPrecision(7));
	} else {
		displayMemory("M: " + memStr);
	}

	logDisplay("doMemory");
	report("doMemory end");
}

function doClear() {
	report("doClear");

	op = "";
	setOpSet(false);
	if (!isNaN(x) && (numString !== "")) {
		n = x;
		numString = String(n);
		displayNumber(numString);
	} else if (n !== 0) {
		x = NaN;
		n = 0;
		numString = String(n);
		displayNumber(numString);
	} else {
		prevOp = "";
		prevN = NaN;
		displayNumber("0");
	}
	numString = "";
	displayPrevOp();

	logDisplay("doClear");
	report("doClear end");
}

function doDelete() {
	numString = numString.substring(0, numString.length - 1);
	n = Number(numString);
	displayNumber(numString);
}

function processOperator(op) {
	switch (op) {
	case "⌈ ⌉":
	case "⌊ ⌋":
	case "x⁻¹":
	case "√x":
	case "x²":
	case "±":
	case "sinh":
	case "cosh":
	case "tanh":
	case "asinh":
	case "acosh":
	case "atanh":
	case "exp":
	case "10ⁿ":
	case " 2ⁿ":
	case "sin":
	case "cos":
	case "tan":
	case "log2":
	case "ln ":
	case "log":
	case "asin":
	case "acos":
	case "atan":
	case "^":
	case "√":
	case "%":
	case "÷":
	case "/":
	case "*":
	case "-":
	case "+":
	case "~":
	case "&":
	case "|":
	case "⊕":
	case "<<":
	case ">>":
	case ">>>":
		doOperator(op);
		break;
	case "mc":
	case "m+":
	case "m-":
	case "mr":
		doMemory(op);
		break;
	case "clr":
		doClear();
		break;
	case "=":
		doEnter();
		break;
	}
}
