/*
    Impedance Chart - Draws an Impedance Chart on a Canvas.
    Copyright  2007-2015 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Impedance Chart - version 2.3
    6 July, 2015
    Copyright  2007-2015 Harry Whitfield
    mailto:g6auc@arrl.net
*/

/*
    impCHART is a global function which draws an impedance chart onto a canvas.

    It takes three parameters and returns a 2d drawing context.

        var ctx = impCHART(canvas, fontSize, fontFamily);

    The first parameter should be the canvas onto which the chart is to be drawn.

    The second parameter is optional. If present, it should be an integer between 9 and 18.
    If omitted or invalid, the fontSize used for labels on the chart is set automatically.

    Labels are omitted if the smaller dimension of the canvas is less than 500.

    The third parameter is optional. If present, it should be a string naming the font family.
    If omitted, the font family 'Lucida Grande' is used.

    The function returns the 2d drawing context, untranslated and unscaled.

        impCHART.reset();

    Resets the chart to its initial state.

        impCHART.setDrawColor(color);

    Sets the color of circles and points. The parameter should be a color string "#rrggbb".

        impCHART.plotCircle(radius);

    Plots a circle centred at the centre of the chart. 0 < radius <= 1.0

        impCHART.plotPoint(x, y);

    Plots a point at position x, y.     x*x + y*y <= 1.0

        impCHART.translate();

    Moves the origin of the context to the middle of the chart.

        impCHART.scale();

    Scales the context so that the radius of the chart is 1.0.
*/

/*jslint bitwise, for, this, white */

/*property
    PI, arc, atan2, beginPath, clip, drawMarker, fill, fillStyle, fillText,
    floor, font, fontScaleX, fontScaleY, getContext, getFullContext, height,
    indexOf, length, lineTo, lineWidth, moveTo, plotCircle, plotPoint,
    prototype, rect, reset, restore, rotate, save, scale, scaleFont,
    setDrawColor, show, stroke, strokeStyle, substring, textBaseline, toFixed,
    translate, width
*/

/////////////////////////// Start of the Canvas Text functions ///////////////////////////
Object.prototype.getFullContext = function (s) {
    'use strict';
    var ctx = this.getContext(s);

    ctx.fontScaleX = 1;
    ctx.fontScaleY = 1;

    ctx.scaleFont = function (scaleX, scaleY) {
        ctx.fontScaleX *= scaleX;
        ctx.fontScaleY *= scaleY;
    };

    ctx.show = function (x, y, s) {
        ctx.save();
        ctx.scale(1 / ctx.fontScaleX, 1 / ctx.fontScaleY);
        ctx.fillText(s, x * ctx.fontScaleX, y * ctx.fontScaleY);
        ctx.restore();
    };

    return ctx;
};
/////////////////////////// End  of  the Canvas Text functions ///////////////////////////


var impCHART = (function () {
    'use strict';
    // An Impedance Chart consists of two families of circles and circular arcs drawn in the (u,v) plane.

    // The R-family circles have their centres at (u = r/(r+1), v = 0) with radius 1/(r+1),
    // and with values of r running from 0.01 to 50. They all lie within the unit circle.

    // The X-family circles have their centres at (u = 1, v = 1/x) with radius |1/x|,
    // and with values of x running from -50 to -0.01, and from 0.01 to 50.
    // They are clipped by the unit circle.

    var width, height, margins, scales, radius,
            dpi, lw0, lw1, lw2, lw3, lw5,
            ctx, drawColor, itself,
            canvasSaved, fontSizeSaved, fontFamilySaved;

    /////////////////////////////// Circle Functions /////////////////////////////////////

    function rdraw(r, f) {                                  // draw R-family circles
        ctx.save();
        ctx.beginPath();
        if ((r % 100) === 0) {
            ctx.lineWidth = lw2;
        }
        if ((r % 1000) === 0) {
            ctx.lineWidth = lw3;
        }
        r = r * f;
        ctx.arc(r / (r + 1), 0, 1 / (r + 1), 0, 2 * Math.PI, false);
        ctx.stroke();
        ctx.restore();
    }

    function rgrid1() {
        var i, f = 0.001;
        ctx.lineWidth = lw1;
        for (i = 50; i <= 150; i += 50) { rdraw(i, f); }
        for (i = 200; i <= 450; i += 50) { rdraw(i, f); }
        for (i = 500; i <= 1000; i += 100) { rdraw(i, f); }
    }

    function rgrid2() {
        var i, f = 0.01;
        ctx.lineWidth = lw1;
        for (i = 120; i <= 180; i += 20) { rdraw(i, f); }
        for (i = 200; i <= 450; i += 50) { rdraw(i, f); }
        for (i = 500; i <= 1000; i += 250) { rdraw(i, f); }
    }

    function rgrid3() {
        var i, f = 0.1;
        ctx.lineWidth = lw1;
        for (i = 150; i <= 200; i += 50) { rdraw(i, f); }
        for (i = 300; i <= 500; i += 100) { rdraw(i, f); }
    }

    function xdraw(r, f) {                                  // draw X-family circles
        ctx.save();
        ctx.beginPath();
        if ((r %  100) === 0) {
            ctx.lineWidth = lw2;
        }
        if ((r % 1000) === 0) {
            ctx.lineWidth = lw3;
        }
        r = r * f;
        ctx.arc(1,  1 / r, 1 / r, Math.PI / 2, 3 * Math.PI / 2, false);
        ctx.stroke();
        ctx.beginPath();
        ctx.arc(1, -1 / r, 1 / r,  Math.PI / 2, 3 * Math.PI / 2, false);
        ctx.stroke();
        ctx.restore();
    }

    function xgrid1() {
        var i, f = 0.001;
        ctx.lineWidth = lw1;
        for (i = 50; i <= 150; i += 50) { xdraw(i, f); }
        for (i = 200; i <= 450; i += 50) { xdraw(i, f); }
        for (i = 500; i <= 1000; i += 100) { xdraw(i, f); }
    }

    function xgrid2() {
        var i, f = 0.01;
        ctx.lineWidth = lw1;
        for (i = 120; i <= 180; i += 20) { xdraw(i, f); }
        for (i = 200; i <= 450; i += 50) { xdraw(i, f); }
        for (i = 500; i <= 1000; i += 250) { xdraw(i, f); }
    }

    function xgrid3() {
        var i, f = 0.1;
        ctx.lineWidth = lw1;
        for (i = 150; i <= 200; i += 50) { xdraw(i, f); }
        for (i = 300; i <= 500; i += 100) { xdraw(i, f); }
    }

    function makeBackground(width, height) {
        ctx.beginPath();
        ctx.rect(0, 0, width, height);
        ctx.fillStyle = "#FFFFFF";  // grey "#9F9F9F"
        ctx.fill();
        ctx.beginPath();
        ctx.rect(9, 9, width - 18, height - 18);
        ctx.fillStyle = "#FFFFFF";  // white
        ctx.fill();
    }

    function drawUnitCircle() {
        ctx.beginPath();
        ctx.lineWidth = lw3;
        ctx.arc(0, 0, 1.0, 0, 2 * Math.PI, false);
        ctx.stroke();
    }

    function drawAxis() {                                   // draw horizontal diameter
        ctx.beginPath();
        ctx.lineWidth = lw3;
        ctx.moveTo(-1, 0);
        ctx.lineTo(1, 0);
        ctx.stroke();
    }

    function drawCenterCircle() {
        ctx.beginPath();
        ctx.lineWidth = lw3;
        ctx.arc(0, 0, 0.015, 0, 2 * Math.PI, false);        // centre circle
        ctx.strokeStyle = "#000000";                        // black
        ctx.stroke();

        ctx.beginPath();
        ctx.arc(0, 0, 0.0075, 0, 2 * Math.PI, false);       // centre point
        ctx.fillStyle = "#000000";                          // black
        ctx.fill();
    }

    function drawChart() {
        ctx.strokeStyle = "#000000";                        // black
        rgrid1();
        rgrid2();
        rgrid3();

        drawCenterCircle();

        ctx.strokeStyle = "#00000";                         // black
        xgrid1();
        xgrid2();
        xgrid3();
    }

    function marker() {
        ctx.beginPath();
        ctx.rect(20, 50, 20, 20);
        ctx.fillStyle = drawColor;                          // drawColor
        ctx.fill();
    }

    //////////////////////////////// Label Functions /////////////////////////////////////

    function swrite1(v) {
        ctx.save();
        v = 0.1 * v;
        var ang = Math.PI - 2 * Math.atan2(v, 1);
        ctx.rotate(-ang);
        ctx.beginPath();
        ctx.show(0.935, -0.05, v.toFixed(1));
        ctx.restore();
    }

    function swrite2(v) {
        ctx.save();
        var ang = Math.PI - 2 * Math.atan2(v, 1);
        ctx.rotate(-ang);
        ctx.beginPath();
        ctx.show(0.95, -0.05, v.toFixed(0));
        ctx.restore();
    }

    function swrite3(v) {
        ctx.save();
        v = 0.1 * v;
        var ang = Math.PI + 2 * Math.atan2(v, 1);
        ctx.rotate(-ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0.99, 0.05);
        ctx.rotate(Math.PI);
        ctx.show(0, 0, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function swrite4(v) {
        ctx.save();
        var ang = Math.PI + 2 * Math.atan2(v, 1);
        ctx.rotate(-ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0.99, 0.05);
        ctx.rotate(Math.PI);
        ctx.show(0, 0, v.toFixed(0));
        ctx.restore();
        ctx.restore();
    }

    function swrite5(v) {
        ctx.save();
        v = 0.1 * v;
        var p = (v - 1) / (v + 1);
        ctx.rotate(-Math.PI / 2);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0, p);
        ctx.show(0.005, -0.045, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function swrite6(v) {
        ctx.save();
        var p = (v - 1) / (v + 1);
        ctx.rotate(-Math.PI / 2);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0, p);
        ctx.show(0.005, -0.045, v.toFixed(0));
        ctx.restore();
        ctx.restore();
    }

    function swrite7(v) {
        ctx.save();
        ctx.translate(1, 1);
        v = 0.1 * v;
        var ang = 3 * Math.PI / 2 - 2 * Math.atan2(1, v + 1);
        ctx.rotate(ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(1, 0);
        ctx.show(-0.065, -0.045, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function swrite8(v) {
        ctx.save();
        ctx.translate(1, -1);
        v = 0.1 * v;
        var ang = Math.PI / 2 + 2 * Math.atan2(1, v + 1);
        ctx.rotate(ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(1, 0);
        ctx.rotate(Math.PI);
        ctx.show(0.005, -0.045, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function swrite9(v) {
        ctx.save();
        ctx.translate(0.5, 0);
        v = 0.1 * v;
        var ang = Math.PI - 2 * Math.atan2(v / 2, 1);
        ctx.rotate(ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0.495, 0.045);
        ctx.rotate(Math.PI);
        ctx.show(0, 0, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function swrite10(v) {
        ctx.save();
        ctx.translate(0.5, 0);
        v = 0.1 * v;
        var ang = Math.PI + 2 * Math.atan2(v / 2, 1);
        ctx.rotate(ang);
        ctx.beginPath();
        ctx.save();
        ctx.translate(0.435, -0.045);
        ctx.show(0, 0, v.toFixed(1));
        ctx.restore();
        ctx.restore();
    }

    function scale3() {
        var i;
        for (i = 1; i <= 9; i += 1) { swrite1(i); }
        for (i = 10; i <= 18; i += 2) { swrite1(i); }
        for (i = 20; i <= 50; i += 10) { swrite1(i); }
        for (i = 10; i <= 20; i += 10) { swrite2(i); }
        for (i = 50; i <= 50; i += 50) { swrite2(i); }
    }

    function scale4() {
        var i;
        for (i = 1; i <= 9; i += 1) { swrite3(i); }
        for (i = 10; i <= 18; i += 2) { swrite3(i); }
        for (i = 20; i <= 50; i += 10) { swrite3(i); }
        for (i = 10; i <= 20; i += 10) { swrite4(i); }
        for (i = 50; i <= 50; i += 50) { swrite4(i); }
    }

    function scale5() {
        var i;
        for (i = 1; i <= 9; i += 1) { swrite5(i); }
        for (i = 10; i <= 18; i += 2) { swrite5(i); }
        for (i = 20; i <= 50; i += 10) { swrite5(i); }
        for (i = 10; i <= 20; i += 10) { swrite6(i); }
        for (i = 50; i <= 50; i += 50) { swrite6(i); }
    }

    function scale7() {
        var i;
        for (i = 2; i <= 10; i += 2) { swrite7(i); }
    }

    function scale8() {
        var i;
        for (i = 2; i <= 10; i += 2) { swrite8(i); }
    }

    function scale9() {
        var i;
        for (i = 2; i <= 10; i += 2) { swrite9(i); }
    }

    function scale10() {
        var i;
        for (i = 2; i <= 10; i += 2) { swrite10(i); }
    }

    //////////////////////////////// User Functions //////////////////////////////////////

    // the following functions are available to the user via the impCHART function

    itself = function (canvas, fontSize, fontFamily, reset) {
        function min(a, b) { if (b < a) { return b; } return a; }

        if (!reset) {
            ctx = canvas.getFullContext("2d");
            canvasSaved = canvas;
            fontSizeSaved = fontSize;
            fontFamilySaved = fontFamily;
        }

        width   = min(canvas.width, canvas.height);
        height  = width;

        margins = 52;   // was 72                           // allowance for two margins
        scales  = 1.0;                                      // maximum radius of scales
        radius = ((width - margins) >> 1) / scales;

        dpi = 288;
        lw0 = (72 / dpi) / radius;
        lw1 = lw0;
        lw2 = 2 * lw0;
        lw3 = 3 * lw0;
        lw5 = 5 * lw0;

        makeBackground(width, height);

        drawColor = "#000000";  // initially black

        marker();

        if (fontSize) { fontSize = Number(fontSize); }

        if (isNaN(fontSize) || (fontSize < 9) || (fontSize > 18)) {
            fontSize = Math.floor(18 * width / 1000);
        } else {
            fontSize = Math.floor(fontSize);
        }

        // "font-style font-variant font-weight font-size/line-height font-family"

        if (!fontFamily || (typeof fontFamily !== "string")) {
            fontFamily = "'Lucida Grande'";
        } else {
            if ((fontFamily[0] === "'") && (fontFamily[fontFamily.length - 1] === "'")) {
                fontFamily = fontFamily.substring(1, fontFamily.length - 1);
            } else if ((fontFamily[0] === '"') && (fontFamily[fontFamily.length - 1] === '"')) {
                fontFamily = fontFamily.substring(1, fontFamily.length - 1);
            }
        }

        if (fontFamily.indexOf(" ") >= 0) {
            fontFamily = "'" + fontFamily + "'";
        }

        ctx.font = String(fontSize) + "px " + fontFamily;
        ctx.textBaseline  = "top";
        ctx.fillStyle = "#000000";                          // black

        ctx.save();
        ctx.translate(width >> 1, height >> 1);             // move origin to middle of page
        ctx.save();
        ctx.scale(radius, radius);                          // scale unit circle to fit page
        drawUnitCircle();                                   // make it the clipping region
        ctx.save();
        ctx.clip();
        drawAxis();
        drawChart();
        ctx.restore();                                      // undo clipping
        ctx.restore();                                      // undo scaling

        if (width >= 500) {                                 // draw labels
            ctx.save();
            ctx.scale(radius, radius);
            ctx.scaleFont(radius, radius);
            scale3();
            scale4();
            scale5();
            scale7();
            scale8();
            scale9();
            scale10();
            ctx.scaleFont(1 / radius, 1 / radius);          // undo font scaling
            ctx.restore();                                  // undo scaling
        }
        ctx.restore();                                      // undo translation

        ctx.font = "14px 'Lucida Grande'";
        ctx.fillStyle = "#000000";          // black
        ctx.fillText("Impedance Chart", 20, 25);    // Add a title

        return ctx;
    };

    itself.reset = function () {
        itself(canvasSaved, fontSizeSaved, fontFamilySaved, true);
    };

    itself.setDrawColor = function (color) {
        drawColor = color;
    };

    itself.drawMarker = marker;

    itself.plotCircle = function (rho) {
        ctx.save();
        ctx.translate(width >> 1, height >> 1);             // move origin to middle of page
        ctx.save();
        ctx.scale(radius, radius);
        ctx.beginPath();
        ctx.lineWidth = lw5;
        ctx.arc(0, 0, rho, 0, 2 * Math.PI, false);
        ctx.strokeStyle = drawColor;
        ctx.stroke();
        ctx.restore();
        ctx.restore();
    };

    itself.plotPoint = function (x, y) {
        ctx.save();
        ctx.translate(width >> 1, height >> 1);             // move origin to middle of page
        ctx.save();
        ctx.scale(radius, radius);
        ctx.beginPath();
        ctx.lineWidth = lw5;
        ctx.arc(x, -y, 0.015, 0, 2 * Math.PI, false);
        ctx.strokeStyle = drawColor;
        ctx.stroke();
        ctx.restore();
        ctx.restore();
    };

    itself.scale = function () {
        ctx.scale(radius, radius);
    };

    itself.translate = function () {
        ctx.translate(width >> 1, height >> 1);
    };

    return itself;
}());

/*
    var canvas = document.getElementById("ImpChart"),   // canvas
        ctx = impCHART(canvas, "10", "Lucida Grande");  // Draw the chart

    impCHART.setDrawColor("#0000FF");   // blue
    impCHART.plotCircle(0.3333333);     // test plots to check orientation
    impCHART.plotPoint(0.3333333, 0.0);

    impCHART.setDrawColor("#FF0000");   // red
    impCHART.plotCircle(0.620174);
    impCHART.plotPoint(0.538461824, 0.307692467);

    ctx.font = "14px 'Lucida Grande'";
    ctx.fillStyle = "#00FF00";          // green
    ctx.fillText("Impedance Chart", 50, 25);    // Add a title
*/
