/*
    NetCalc - An RF Impedance Calculator
    Copyright 2000-2015 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    NetCalc - version 2.2
    6 July, 2015
    Copyright 2000-2015 Harry Whitfield
    g6auc@arrl.net
*/

/*jslint browser */

/*global Complex, precision, freq, FMultiplier, velfac, resist, RMultiplier, induct,
         LMultiplier, react, XMultiplier, capacit, CMultiplier, Preg, theXreg, Nreg,
         Z0reg, Rreg, degreeSign, theZreg, theYreg, Factor, Mreg, Tlength,
         ohmSign, checkChart, newTextArea, marker, eprint,
         RMenu, LMenu, XMenu, CMenu, FMenu, alert, window
*/

/* properties
    INFINITY, LOG10E, PI, abs, appendChild, arg, atan, cadd, ccosh, cdiv, cmul,
    conj, cos, createElement, csinh, csub, log, modsqr, modulus, one, open, pop,
    pow, push, removeChild, sdiv, sin, smul, splice, split, sqrt, text, title,
    toFixed, toPrecision, toString, value, x, y, zero
*/

var Results;

// -- Functions to compute scaled values from pulldown menus -------------

function frequency() {
    'use strict';
    return parseFloat(freq.value) * Math.pow(10, FMultiplier);
}

function velocity() {
    'use strict';
    return parseFloat(velfac.value);
}
function resistance() {
    'use strict';
    return parseFloat(resist.value) * Math.pow(10, RMultiplier);
}

function inductance() {
    'use strict';
    return parseFloat(induct.value) * Math.pow(10, LMultiplier);
}

function reactance() {
    'use strict';
    return parseFloat(react.value) * Math.pow(10, XMultiplier);
}

function capacitance() {
    'use strict';
    return parseFloat(capacit.value) * Math.pow(10, CMultiplier);
}

// -- Functions to perform electrical network calculations ---------------

// equivalent impedance of a in parallel with b
function par(a, b) {
    'use strict';
    var z;

    if ((a.modsqr() === 0) || (b.modsqr() === 0)) {
        return Complex.zero;
    }
    z = Complex.cadd(a, b);
    if (z.modsqr() === 0) {
        return Complex.INFINITY;
    }
    return Complex.cdiv(Complex.cmul(a, b), z);
}

// impedance a expressed in parallel form {R, X} such that a = {R, 0} || {0, X} as string
function parform(a) {
    'use strict';
    var ms = a.x * a.x + a.y * a.y,
        tmp1,
        tmp2;

    if (ms === 0) {
        return "0,0";
    }
    tmp1 = ms / a.x;
    tmp2 = ms / a.y;
    return tmp1.toFixed(precision) + "," + tmp2.toFixed(precision);
}

// impedance a expressed in parallel form "R || X" such that a = {R, 0} || {0, X} as string
function pform(a) {
    'use strict';
    var ms = a.x * a.x + a.y * a.y,
        tmp1,
        tmp2;

    if (ms === 0) {
        return "0 || 0";
    }
    tmp1 = ms / a.x;
    tmp2 = ms / a.y;
    return tmp1.toFixed(precision) + " || " + tmp2.toFixed(precision);
}

// -- Functions to perform transmission line calculations ----------------

// Reflection Coefficient rho = ((Z1 / Z0) - 1) / ((Z1 / Z0) + 1)
function rho(Z1, Z0) {
    'use strict';
    var z = Complex.cdiv(Z1, Z0);

    return Complex.cdiv(Complex.csub(z, Complex.one), Complex.cadd(z, Complex.one));
}

// Voltage Standing Wave Ratio VSWR = (1 + |rho|) / (1 - |rho|)
function vswr(rho) {
    'use strict';
    var r = rho.modulus();

    return (1 + r) / (1 - r);
}

// log to the base 10 in terms of log to the base e
function log10(x) {
    'use strict';
    return Math.LOG10E * Math.log(x);
}

// Return Loss = -20 * log(|rho|) dB = -10 * log(|rho|^2) dB
function returnLoss(rho) {
    'use strict';
    return -10 * log10(rho.modsqr());
}

// input impedance of a lossless transmission line of characteristic impedance Z0
// and length x wavelengths which is terminated in a load impedance Z1
function zin(Z1, Z0, x) {
    'use strict';
    var C = new Complex(Math.cos(2 * Math.PI * x), 0),
        S = new Complex(0, Math.sin(2 * Math.PI * x)),
        Z = Complex.cdiv(Z1, Z0);

    return Complex.cmul(Z0, Complex.cdiv(Complex.cadd(Complex.cmul(Z, C), S), Complex.cadd(C, Complex.cmul(S, Z))));
}

// input impedance of a lossy transmission line of characteristic impedance Z0
// and attenuation factor alpha (nepers/metre)
// and length x wavelengths which is terminated in a load impedance Z1
function zin2(Z1, Z0, x, alpha) {
    'use strict';
    var c = 299792458, // speed of light in m/s
        v = velocity(),
        lamda = (c * v) / frequency(),
        length = x * lamda,
        arg = new Complex(alpha * length, 2 * Math.PI * x),
        C = Complex.ccosh(arg),
        S = Complex.csinh(arg),
        Z = Complex.cdiv(Z1, Z0);

    return Complex.cmul(Z0, Complex.cdiv(Complex.cadd(Complex.cmul(Z, C), S), Complex.cadd(C, Complex.cmul(S, Z))));
}

// length of transmission line of characteristic impedance Z0 having
// the same reactance at frequency f as the impedance Z1.
function equivLength(Z1, Z0, f, v) {
    'use strict';
    var Z = Complex.cdiv(Z1, Z0),
        x = Math.atan(Z.y);

    if (x < 0) {
        x += Math.PI;
    }
    return x * v / (2 * Math.PI * f);
}

function prefix(v) {
    'use strict';
    var x = Math.abs(v);

    if (x >= 1000000) {
        return "M";
    }
    if (x >= 1000) {
        return "k";
    }
    if (x >= 1) {
        return "";
    }
    if (x >= 0.001) {
        return "m";
    }
    if (x >= 0.000001) {
        return "µ";
    }
    if (x >= 0.000000001) {
        return "n";
    }
    return "p";
}

function multiplier(v) {
    'use strict';
    var x = Math.abs(v);

    if (x >= 1000000) {
        return 0.000001;
    }
    if (x >= 1000) {
        return 0.001;
    }
    if (x >= 1) {
        return 1.0;
    }
    if (x >= 0.001) {
        return 1000;
    }
    if (x >= 0.000001) {
        return 1000000;
    }
    if (x >= 0.000000001) {
        return 1000000000;
    }
    return 1000000000000;
}

// -- Functions used for button actions ----------------------------------

var theYstack = [];     // overflow stack for theYreg

function toComplex(s) {
    'use strict';
    var item = s.split(",");

    return new Complex(Number(item[0]), Number(item[1]));
}

function update() {                 // update display - used after (almost) every action
    'use strict';
    var rh, theta, tmp;

    Preg.value = pform(toComplex(theXreg.value));
    Nreg.value = (Complex.cdiv(toComplex(theXreg.value), toComplex(Z0reg.value))).toString();

    rh = rho(toComplex(theXreg.value), toComplex(Z0reg.value));
    Rreg.value = rh.modulus().toFixed(precision);
    theta = rh.arg();
    if (isNaN(theta)) {
        Rreg.value += " @ UNDEFINED" + degreeSign;
    } else {
        tmp = 180 * theta / Math.PI;
        Rreg.value += " @ " + tmp.toFixed(precision) + degreeSign;
    }
    theZreg.value = "VSWR=" + vswr(rh).toFixed(precision) + " RL=" + returnLoss(rh).toFixed(precision) + "dB";

    eprint("\tY:   " + theYreg.value);
    eprint("\tX:   " + theXreg.value);
    eprint("\tP:   " + Preg.value);
    eprint("\tZ0:  " + Z0reg.value);
    eprint("\tN:   " + Nreg.value);
    eprint("\trho: " + Rreg.value);
    eprint("\tZ:   " + theZreg.value);
}

function showMod(s) {                   // display s in polar form in theZreg
    'use strict';
    var X = toComplex(s), theta, tmp;

    theZreg.value = X.modulus().toFixed(precision);
    theta = X.arg();
    if (isNaN(theta)) {
        theZreg.value += " @ UNDEFINED" + degreeSign;
    } else {
        tmp = 180 * theta / Math.PI;
        theZreg.value += " @ " + tmp.toFixed(precision) + degreeSign;
    }
    eprint("\tZ:   " + theZreg.value);
}

function showVSWR() {
    'use strict';
    var rh = rho(toComplex(theXreg.value), toComplex(Z0reg.value));

    eprint("\nW Button");
    theZreg.value = "VSWR=" + vswr(rh).toFixed(precision) + " RL=" + returnLoss(rh).toFixed(precision) + "dB";
    eprint("\tZ:   " + theZreg.value);
}

function btnC(altKey) {         // pop
    'use strict';
    if (altKey) {
        eprint("\nC Button with alt-key");
        theYstack.splice(0);
        theXreg.value = "0,0";
        theYreg.value = "0,0";
    } else {
        eprint("\nC Button");
        theXreg.value = theYreg.value;
        theYreg.value = theYstack.pop() || "0,0";
    }
    update();
}

function btnJ() {
    'use strict';
    eprint("\nJ Button");
    theXreg.value = toComplex(theXreg.value).conj().toString();
    update();
}

function btnE() {
    'use strict';
    var XX = theXreg.value;

    eprint("\nE Button");
    theXreg.value = theYreg.value;
    theYreg.value = XX;
    update();
}

function btnQ() {
    'use strict';
    var X = toComplex(theXreg.value);

    eprint("\nQ Button");
    theZreg.value = "Q=" + Math.abs(X.y / X.x).toFixed(precision);
    eprint("\tZ:   " + theZreg.value);
}

function btnParallel() {        // pop
    'use strict';
    eprint("\n|| Button");
    theXreg.value = par(toComplex(theXreg.value), toComplex(theYreg.value)).toString();
    theYreg.value = theYstack.pop() || "0,0";
    update();
}

function btnSeries() {          // pop
    'use strict';
    eprint("\n+ Button");
    theXreg.value = Complex.cadd(toComplex(theXreg.value), toComplex(theYreg.value)).toString();
    theYreg.value = theYstack.pop() || "0,0";
    update();
}

function btnXL() {              // push
    'use strict';
    eprint("\nInput R(" + resist.value + RMenu.value + ") + L(" + induct.value + LMenu.value + ") at F(" + freq.value + FMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    theXreg.value = (new Complex(resistance(), 2 * Math.PI * frequency() * inductance())).toString();
    update();
}

function btnXX() {              // push
    'use strict';
    eprint("\nInput R(" + resist.value + RMenu.value + ") + X(" + react.value + XMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    theXreg.value = (new Complex(resistance(), reactance())).toString();
    update();
}

function btnXC() {              // push
    'use strict';
    eprint("\nInput R(" + resist.value + RMenu.value + ") + C(" + capacit.value + CMenu.value + ") at F(" + freq.value + FMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    theXreg.value = (new Complex(resistance(), -1 / (2 * Math.PI * frequency() * capacitance()))).toString();
    update();
}

function btnXLP() {             // push
    'use strict';
    var RR, XX;

    eprint("\nInput R(" + resist.value + RMenu.value + ") || L(" + induct.value + LMenu.value + ") at F(" + freq.value + FMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    RR = new Complex(resistance(), 0);
    XX = new Complex(0, 2 * Math.PI * frequency() * inductance());
    theXreg.value = par(RR, XX).toString();
    update();
}

function btnXXP() {             // push
    'use strict';
    var RR, XX;

    eprint("\nInput R(" + resist.value + RMenu.value + ") || X(" + react.value + XMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    RR = new Complex(resistance(), 0);
    XX = new Complex(0, reactance());
    theXreg.value = par(RR, XX).toString();
    update();
}

function btnXCP() {             // push
    'use strict';
    var RR, XX;

    eprint("\nInput R(" + resist.value + RMenu.value + ") || C(" + capacit.value + CMenu.value + ") at F(" + freq.value + FMenu.value + ")");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    RR = new Complex(resistance(), 0);
    XX = new Complex(0, -1 / (2 * Math.PI * frequency() * capacitance()));
    theXreg.value = par(RR, XX).toString();
    update();
}

function btnmul() {
    'use strict';
    var factor = parseFloat(Factor.value);

    eprint("\n* Button with Factor " + Factor.value);

    if (isNaN(factor)) {
        eprint("\tInvalid Factor");
        alert("Invalid Factor");
    } else {
        theXreg.value = Complex.smul(factor, toComplex(theXreg.value)).toString();
        update();
    }
}

function btndiv() {
    'use strict';
    var factor = parseFloat(Factor.value);

    eprint("\n/ Button with Factor " + Factor.value);

    if (isNaN(factor)) {
        eprint("\tInvalid Factor");
        alert("Invalid Factor");
    } else {
        theXreg.value = Complex.sdiv(factor, toComplex(theXreg.value)).toString();
        update();
    }
}

function btnL() {
    'use strict';
    var c = 299792458,      // speed of light in m/s
        tmp;

    eprint("\nL Button at F(" + freq.value + FMenu.value + ") with V(" + velfac.value + ")");
    tmp = 1000 * equivLength(toComplex(theXreg.value), toComplex(Z0reg.value), frequency(), c * velocity());
    theZreg.value = "Length=" + tmp.toFixed(1) + "mm";

    eprint("\tX:   " + theXreg.value);
    eprint("\tP:   " + Preg.value);
    eprint("\tZ0:  " + Z0reg.value);
    eprint("\tZ:   " + theZreg.value);
}

function menuRn(i) {                // push
    'use strict';
    eprint("\nR" + i + " Button");
    theYstack.push(theYreg.value);
    theYreg.value = theXreg.value;
    theXreg.value = Mreg[i];
    update();
}

function menuQn(i) {
    'use strict';
    eprint("\n?" + i + " Button");
    theZreg.value = Mreg[i];
    eprint("\tZ:   " + theZreg.value);
}

function menuMn(i) {
    'use strict';
    eprint("\nM" + i + " Button");
    Mreg[i] = theXreg.value;
    eprint("\tM" + i + ":  " + Mreg[i]);
}

function btnHelp(path) {
    'use strict';
    window.open(path);
}

function btnX() {
    'use strict';
    var X = toComplex(theXreg.value),
        it = X.y,
        tmp;

    eprint("\nX Button at F(" + freq.value + FMenu.value + ")");

    if (it >= 0) {
        it = it / (2 * Math.PI * frequency());
        tmp = it * multiplier(it);
        theZreg.value = "Lx=" + tmp.toFixed(precision) + prefix(it) + "H";
    } else {
        it = -1 / (2 * Math.PI * frequency() * it);
        tmp = it * multiplier(it);
        theZreg.value = "Cx=" + tmp.toFixed(precision) + prefix(it) + "F";
    }

    it = toComplex(parform(X)).y;

    if (isFinite(it)) {
        if (it >= 0) {
            it = it / (2 * Math.PI * frequency());
            tmp = it * multiplier(it);
            theZreg.value += "  Lp=" + tmp.toFixed(precision) + prefix(it) + "H";
        } else {
            it = -1 / (2 * Math.PI * frequency() * it);
            tmp = it * multiplier(it);
            theZreg.value += "  Cp=" + tmp.toFixed(precision) + prefix(it) + "F";
        }
    } else {
        theZreg.value += "  Xp=Infinite";
    }

    eprint("\tX:   " + theXreg.value);
    eprint("\tP:   " + Preg.value);
    eprint("\tZ:   " + theZreg.value);
}

function btnT(optionKey) {
    'use strict';
    var it,
        c,
        Z0 = toComplex(Z0reg.value),
        Z1 = toComplex(theXreg.value),
        x = parseFloat(Tlength.value),
        v = 0;

    eprint("\nT Button at F(" + freq.value + FMenu.value + ") with V(" + velfac.value + ")");

    if (isNaN(x)) {
        eprint("\tInvalid Tlength");
        alert("Invalid Tlength");
    } else {
        if (x >= 1) {           //   x in mm
            c = 299792458;      // speed of light in m/s
            v = velocity();
            x = 0.001 * x * frequency() / (c * v);
        }

        if (optionKey) {            // lossy transmission line
            it = parseFloat(Factor.value);
            if (isNaN(it)) {
                eprint("\tInvalid Factor");
                alert("Invalid Factor");
            } else {
                theXreg.value = zin2(Z1, Z0, x, it).toString();
                update();
            }
        } else {                    // lossless transmission line
            theXreg.value = zin(Z1, Z0, x).toString();
            update();
        }
    }
}

function say(optionKey, x, omega) {
    'use strict';
    var it = 0,
        tmp;

    if (optionKey) {
        it = x;
        tmp = it * multiplier(it);
        Results.value += "reactance is " + tmp.toPrecision(6) + prefix(it) + ohmSign + "\n";
    } else {
        if (x < 0) {
            it = -1 / (x * omega);
            tmp = it * multiplier(it);
            Results.value += "capacitance is " + tmp.toPrecision(6) + prefix(it) + "F\n";
        } else {
            it = x / omega;
            tmp = it * multiplier(it);
            Results.value += "inductance     is " + tmp.toPrecision(6) + prefix(it) + "H\n";
        }
    }
}

function computeLS(optionKey, omega, R0, R, X) {
    'use strict';
    var A, G, B, P, X1, B2, X2;

    Results = newTextArea(50, 35, 46, 15, "Test Data", 10, "true");
    Results.title = "Results of L-Section calculation.\nClick on LS button to dismiss.";

    Results.value = "";

    A = R * R + X * X;
    G = R / A;
    B = -X / A;

    Results.value += "Shunt arm parallel to load\n";

    P = 1 / (G * R0);

    X1 = 0;
    B2 = 0;
    X2 = 0;

    if (P < 1) {
        Results.value += "No solution\n";
    } else {
        A = Math.sqrt(P - 1);
        X1 = R0 * A;
        B2 = -A / (P * R0);

        Results.value += "First solution\n  Series arm ";
        say(optionKey, -X1, omega);
        Results.value += "  Shunt  arm ";
        say(optionKey, -1 / (B2 - B), omega);
        Results.value += "Second solution\n  Series arm ";
        say(optionKey, X1, omega);
        Results.value += "  Shunt  arm ";
        say(optionKey, -1 / (-B2 - B), omega);
    }
    Results.value += "\n";

    Results.value += "Shunt arm parallel to generator\n";

    P = R0 / R;

    if (P < 1) {
        Results.value += "No solution\n";
    } else {
        A = Math.sqrt(P - 1);
        X1 = R * A;
        X2 = P * R / A;
        Results.value += "First solution\n  Series arm ";
        say(optionKey, -X1 - X, omega);
        Results.value += "  Shunt  arm ";
        say(optionKey, X2, omega);
        Results.value += "Second solution\n  Series arm ";
        say(optionKey, X1 - X, omega);
        Results.value += "  Shunt  arm ";
        say(optionKey, -X2, omega);
    }
}

var btnLSToggle = false;
var _limbo = document.createElement('div');

function deleteElement(ele) {
    'use strict';
    _limbo.appendChild(ele);
    _limbo.removeChild(ele);
}

function btnLS(optionKey) {
    'use strict';
    var Z0, X, omega;

    btnLSToggle = !btnLSToggle;

    if (btnLSToggle) {

        eprint("\nLS Button at F(" + freq.value + FMenu.value + ")");
        eprint("X:   " + theXreg.value);
        eprint("P:   " + Preg.value);
        eprint("Z0:  " + Z0reg.value);

        Z0 = toComplex(Z0reg.value);
        X = toComplex(theXreg.value);
        omega = 2 * Math.PI * frequency();
        computeLS(optionKey, omega, Z0.x, X.x, X.y);


        eprint("Results:\n" + Results.value);
    } else {
        deleteElement(Results);

    }
}
