/*
	NetCalc - An RF Impedance Calculator
	Copyright 2000-2014 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

	NetCalc - version 2.0
	21 December, 2014
	Copyright 2000-2014 Harry Whitfield
	g6auc@arrl.net
*/

// -- Functions to perform complex arithmetic ----------------------------

/*jslint browser, multivar, this */

/*global precision */

/*property
    INFINITY, NAN, PI, arg, atan, cadd, ccosh, ccosh2, cdiv, cexp, cmul, cneg,
    conj, cos, csinh, csinh2, csub, ctanh, ctanh2, exp, i, im, modsqr, modulus,
    one, prototype, re, sdiv, sin, smul, sqrt, toFixed, toString, x, y, zero
*/

function sinh(x) {
	"use strict";
	return 0.5 * (Math.exp(x) - Math.exp(-x));
}

function cosh(x) {
	"use strict";
	return 0.5 * (Math.exp(x) + Math.exp(-x));
}

function tanh(x) {
	"use strict";
	return sinh(x) / cosh(x);
}

function Complex(real, imaginary) {					// constructor
	"use strict";
	this.x = real;
	this.y = imaginary;
}

Complex.zero = new Complex(0, 0);
Complex.one = new Complex(1, 0);
Complex.i = new Complex(0, 1);
Complex.NAN = new Complex(NaN, NaN);
Complex.INFINITY = new Complex(Infinity, Infinity);

Complex.prototype.modulus = function () {			// modulus of complex number
	"use strict";
	return Math.sqrt(this.x * this.x + this.y * this.y);
};

Complex.prototype.arg = function () {				// argument of complex number
	"use strict";
													// result in -π < arg <= π
	if (this.x === 0) {
		if (this.y === 0) {
			return NaN;
		}
		if (this.y > 0) {
			return 0.5 * Math.PI;
		}
		return -0.5 * Math.PI;
	}
	if (this.x > 0) {
		return Math.atan(this.y / this.x);
	}
	if (this.y >= 0) {
		return Math.atan(this.y / this.x) + Math.PI;
	}
	return Math.atan(this.y / this.x) - Math.PI;
};

Complex.prototype.modsqr = function () {			// modulus squared of complex number
	"use strict";
	return this.x * this.x + this.y * this.y;
};

Complex.prototype.cneg = function () {				// negative of complex number
	"use strict";
	return new Complex(-this.x, -this.y);
};

Complex.prototype.conj = function () {				// complex conjugate of complex number
	"use strict";
	return new Complex(this.x, -this.y);
};

Complex.prototype.toString = function () {			// string representation of complex number
	"use strict";
	return this.x.toFixed(precision) + "," + this.y.toFixed(precision);
};

Complex.prototype.re = function () {				// real part of complex number
	"use strict";
	return this.x;
};

Complex.prototype.im = function () {				// imaginary part of complex number
	"use strict";
	return this.y;
};

Complex.cadd = function (a, b) {					// complex sum of a and b
	"use strict";
	return new Complex(a.x + b.x, a.y + b.y);
};

Complex.csub = function (a, b) {					// complex difference of a and b
	"use strict";
	return new Complex(a.x - b.x, a.y - b.y);
};

Complex.cmul = function (a, b) {					// complex product of a and b
	"use strict";
	return new Complex(a.x * b.x - a.y * b.y, a.x * b.y + a.y * b.x);
};

Complex.cdiv = function (a, b) {					// complex quotient of a and b
	"use strict";
	var ms = b.modsqr();
	if (ms === 0) {
    	if (a.modsqr() === 0) {
    		return Complex.NAN;
    	}
    	return Complex.INFINITY;
    }
    return Complex.smul(1 / ms, Complex.cmul(a, b.conj()));
};

Complex.smul = function (s, a) {					// product of a and scalar s
	"use strict";
	return new Complex(s * a.x, s * a.y);
};

Complex.sdiv = function (s, a) {					// quotient of a and scalar s
	"use strict";
	if (s === 0) {
    	if (a.modsqr() === 0) {
    		return Complex.NAN;
    	}
    	return Complex.INFINITY;
    }
	return new Complex(a.x / s, a.y / s);
};

Complex.cexp = function (a) {						// complex exponential of complex a
	"use strict";
	var r = Math.exp(a.x);
    return new Complex(r * Math.cos(a.y), r * Math.sin(a.y));
};

Complex.csinh = function (a) {						// complex sinh of complex a
	"use strict";
    return Complex.smul(0.5, Complex.csub(Complex.cexp(a), Complex.cexp(a.cneg())));
};

Complex.csinh2 = function (a) {						// alternate complex sinh of complex a
	"use strict";
	return new Complex(sinh(a.x) * Math.cos(a.y), cosh(a.x) * Math.sin(a.y));
};

Complex.ccosh = function (a) {						// complex cosh of complex a
	"use strict";
	return Complex.smul(0.5, Complex.cadd(Complex.cexp(a), Complex.cexp(a.cneg())));
};

Complex.ccosh2 = function (a) {						// alternate complex cosh of complex a
	"use strict";
	return new Complex(cosh(a.x) * Math.cos(a.y), sinh(a.x) * Math.sin(a.y));
};

Complex.ctanh = function (a) {						// complex tanh of complex a
	"use strict";
	return Complex.cdiv(Complex.csinh(a), Complex.ccosh(a));
};

Complex.ctanh2 = function (a) {						// alternate complex tanh of complex a
	"use strict";
	var ch = cosh(a.x),
		sh = sinh(a.x),
  		c = Math.cos(a.y),
		s = Math.sin(a.y),
  		d = ch * ch * c * c + sh * sh * s * s;

	return new Complex(sh * ch / d, s * c / d);
};
/*
function runComplexTest() {
	'use strict';

	eprint("zero=" + Complex.zero);
	eprint("one=" + Complex.one);
	eprint("i=" + Complex.i);
	eprint("NAN=" + Complex.NAN);
	eprint("INFINITY=" + Complex.INFINITY);

	var a = new Complex(1, 1);
	var b = new Complex(3, 4);

	eprint("a=" + a);
	eprint("b=" + b);

	eprint("mod a=" + a.modulus());
	eprint("mod b=" + b.modulus());

	eprint("mod NAN=" + Complex.NAN.modulus());
	eprint("mod INFINITY=" + Complex.INFINITY.modulus());

	eprint("arg a=" + a.arg());
	eprint("arg b=" + b.arg());

	eprint("arg zero=" + Complex.zero.arg());
	eprint("arg one=" + Complex.one.arg());
	eprint("arg i=" + Complex.i.arg());

	eprint("arg NAN=" + Complex.NAN.arg());
	eprint("arg INFINITY=" + Complex.INFINITY.arg());

	eprint("modsqr a=" + a.modsqr());
	eprint("modsqr b=" + b.modsqr());

	eprint("modsqr NAN=" + Complex.NAN.modsqr());
	eprint("modsqr INFINITY=" + Complex.INFINITY.modsqr());

	eprint("cneg b=" + b.cneg());
	eprint("conj b=" + b.conj());
	eprint("toString b=" + b.toString());
	eprint("re b=" + b.re());
	eprint("im b=" + b.im());

	eprint("cadd(a,b)=" + Complex.cadd(a, b));
	eprint("csub(a,b)=" + Complex.csub(a, b));
	eprint("cmul(a,b)=" + Complex.cmul(a, b));
	eprint("cdiv(a,b)=" + Complex.cdiv(a, b));

	eprint("cdiv(a,Complex.zero)=" + Complex.cdiv(a, Complex.zero));
	eprint("cdiv(Complex.zero,Complex.zero)=" + Complex.cdiv(Complex.zero, Complex.zero));

	eprint("smul(3,b)=" + Complex.smul(3, b));
	eprint("cdiv(3,b)=" + Complex.sdiv(3, b));

	eprint("smul(0,b)=" + Complex.smul(0, b));
	eprint("cdiv(0,b)=" + Complex.sdiv(0, b));
	eprint("cdiv(0,Complex.zero)=" + Complex.sdiv(0, Complex.zero));

	eprint("cexp(Complex.one)=" + Complex.cexp(Complex.one));
	eprint("cexp(Complex.i * PI)=" + Complex.cexp(Complex.smul(Math.PI, Complex.i)));
	eprint("cexp(Complex.i)=" + Complex.cexp(Complex.i));

	eprint("csinh a=" + Complex.csinh(a));
	eprint("csinh2 a=" + Complex.csinh2(a));

	eprint("ccosh a=" + Complex.ccosh(a));
	eprint("ccosh2 a=" + Complex.ccosh2(a));

	eprint("ctanh a=" + Complex.ctanh(a));
	eprint("ctanh2 a=" + Complex.ctanh2(a));

	eprint("csinh b=" + Complex.csinh(b));
	eprint("csinh2 b=" + Complex.csinh2(b));

	eprint("ccosh b=" + Complex.ccosh(b));
	eprint("ccosh2 b=" + Complex.ccosh2(b));

	eprint("ctanh b=" + Complex.ctanh(b));
	eprint("ctanh2 b=" + Complex.ctanh2(b));

	eprint("exp 5=" + Math.exp(5));
	eprint("exp 10=" + Math.exp(10));
	eprint("exp 20=" + Math.exp(20));
	eprint("exp 30=" + Math.exp(30));
	eprint("exp 100=" + Math.exp(100));
	eprint("exp 500=" + Math.exp(500));
}
*/
