/*
    InterCircles - computes properties of intersecting circles.
    Copyright © 2019 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.    See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

    InterCircles - browser version 1.0
    14 April, 2019
    Copyright 2019 Harry Whitfield
    mailto:g6auc@arrl.net
 */

/*jslint browser, this */

/*global paramOne, paramTwo, paramThree, paramFour, paramFive,
    headingOne, headingTwo, headingThree, headingFour, headingFive, clearData,
	maxWidth, Menu, newCanvas, checkbox
*/

/*property
    PI, abs, arc, beginPath, checked, ctx, fill, fillStyle, focus, height,
    inRange, isNaN, keyCode, lineTo, lineWidth, moveTo, onchange, onkeydown,
    onkeypress, opacity, preventDefault, prototype, rect, restore, save, scale,
    sqrt, stopPropagation, stroke, strokeStyle, style, toFixed, translate,
    value, which, width
*/

// map custom names onto model names
var xOne = paramOne;
var radiusOne = paramTwo;
var xTwo = paramThree;
var radiusTwo = paramFour;
var results = paramFive;

var xOneHeading = headingOne;
var radiusOneHeading = headingTwo;
var xTwoHeading = headingThree;
var radiusTwoHeading = headingFour;
var resultsHeading = headingFive;

var scaleMenu = Menu[0];

var gErrMess = "Invalid data";

var skinPref = false;

Number.prototype.inRange = function (a, b) {
    return (this >= a) && (this <= b);
};

/////////////////////////////////////// Canvas Code //////////////////////////////////////

var canvasWidth = 1000;
var canvasHeight = 500;
var y0 = canvasHeight / 2;
var cScale = 100;
var x0 = canvasWidth / 2;

var canvas = newCanvas(maxWidth, 0, canvasWidth, canvasHeight, "Resources/bg.png", 1, 1.0);
var ctx = canvas.ctx;

function makeBackground(width, height) {
	ctx.beginPath();
	ctx.rect(0, 0, width, height);
	ctx.fillStyle = "#9F9F9F";	// grey
	ctx.fill();
	ctx.beginPath();
	ctx.rect(9, 9, width - 18, height - 18);
	ctx.fillStyle = "#FFFFFF";	// white
	ctx.fill();
}

function axis(x1, x2) {
	ctx.save();
	ctx.strokeStyle = "black";
	ctx.lineWidth = 1;
	ctx.beginPath();
	ctx.moveTo(x1, y0);
	ctx.lineTo(x2, y0);
	ctx.stroke();
	ctx.restore();
}

function reset() {
	makeBackground(canvas.width, canvas.height);
	axis(10, canvasWidth - 10);
}

function drawUnitCircle() {
	ctx.beginPath();
	ctx.arc(0, 0, 1.0, 0, 2 * Math.PI, false);
	ctx.stroke();
}

function drawCircle(r) {
	ctx.save();
	ctx.scale(r, r);
	ctx.lineWidth = 1 / r;
	drawUnitCircle();
	ctx.restore();
}

function drawPoint(r) {
	ctx.save();
	ctx.scale(r, r);
	ctx.lineWidth = 1 / r;
	drawUnitCircle();
	ctx.restore();
}

function blue(x, y, r) {
	ctx.save();
	ctx.translate(x + x0, y + y0);
	ctx.strokeStyle = "blue";
	drawCircle(r);
	drawPoint(3);
	ctx.restore();
}

function red(x, y, r) {
	ctx.save();
	ctx.translate(x + x0, y + y0);
	ctx.strokeStyle = "red";
	drawCircle(r);
	drawPoint(3);
	ctx.restore();
}

function plotOne(x, r) {
	ctx.save();
	red(x * cScale, 0, r * cScale);
	ctx.restore();
}

function plotTwo(x, r) {
	ctx.save();
	blue(x * cScale, 0, r * cScale);
	ctx.restore();
}

/////////////////////////////////// End of Canvas Code ///////////////////////////////////

/*
	Intersecting Circles

	Two circles with centres (x1, 0) and (x2, 0) with radii r1 and r2.

	xc(x1, x2, r1, r2) returns x0 the x coordinate of the intersection points (if any).

	yc(x0, x2, r2) returns the positive y coordinate of the intersection points.

	status(x1, x2, r1, r2) returns information of how the two circles interact.

	chordLength(x1, x2, r1, r2) returns the (maximum) diameter of a lens made from
	the intersecting circles
*/

function xc(x1, x2, r1, r2) {
	return (x1 * x1 - x2 * x2 - r1 * r1 + r2 * r2) / (2 * (x1 - x2));
}

function yc(x0, x2, r2) {
	var d = x0 - x2;

	return Math.sqrt(r2 * r2 - d * d);
}

function outside(x1, x2, r1, r2) {
	var v2 = Math.abs(x1 - x2);
	var v3 = r1 + r2;

	return v2 > v3;
}

function touching1(x1, x2, r1, r2) {
	var v2 = Math.abs(x1 - x2);
	var v3 = r1 + r2;

	return v2 === v3;
}

function intersecting(x1, x2, r1, r2) {
	var v1 = Math.abs(r1 - r2);
	var v2 = Math.abs(x1 - x2);
	var v3 = r1 + r2;

	return (v1 < v2) && (v2 < v3);
}

function touching2(x1, x2, r1, r2) {
	var v1 = Math.abs(r1 - r2);
	var v2 = Math.abs(x1 - x2);

	return v1 === v2;
}

function inside(x1, x2, r1, r2) {
	var v1 = Math.abs(r1 - r2);
	var v2 = Math.abs(x1 - x2);

	return v1 > v2;
}

function status(x1, x2, r1, r2) {
	if (outside(x1, x2, r1, r2)) {
		return "outside";
	}
	if (touching1(x1, x2, r1, r2)) {
		return "touching1";
	}
	if (intersecting(x1, x2, r1, r2)) {
		return "intersecting";
	}
	if (touching2(x1, x2, r1, r2)) {
		return "touching2";
	}
	if (inside(x1, x2, r1, r2)) {
		return "inside";
	}
	return "unresolved";
}

function chordLength(x1, x2, r1, r2) {
	var x00 = xc(x1, x2, r1, r2);
	var y00 = yc(x00, x2, r2);

	return 2 * y00;
}

function process() {
    var x1 = parseFloat(xOne.value);
    var r1 = parseFloat(radiusOne.value);
    var x2 = parseFloat(xTwo.value);
    var r2 = parseFloat(radiusTwo.value);

    var statusValue;
    var chordValue;

    if (Number.isNaN(x1)) {
        results.value = "Invalid Circle One Center";
        return;
    }

    if (Number.isNaN(r1) || (r1 <= 0)) {
        results.value = "Invalid Radius One Value";
        return;
    }

    if (Number.isNaN(x2)) {
        results.value = "Invalid Circle Two Center";
        return;
    }

    if (Number.isNaN(r2) || (r2 <= 0)) {
        results.value = "Invalid Radius Two Value";
        return;
    }

    statusValue = status(x1, x2, r1, r2);
    chordValue = chordLength(x1, x2, r1, r2);

    results.value = statusValue + "  " + (
    	Number.isNaN(chordValue)
    	? "undefined"
    	: chordValue.toFixed(3)
    );

	if (checkbox.checked) {
    	reset();
    	plotOne(x1, r1);
    	plotTwo(x2, r2);
    	canvas.style.opacity = "1.0";
    } else {
    	canvas.style.opacity = "0.0";
    }
}

checkbox.onchange = function () {
	if (checkbox.checked) {
		process();
	} else {
		canvas.style.opacity = "0.0";
	}
};

xOne.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

xOne.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		radiusOne.focus();
	}
};

radiusOne.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

radiusOne.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		xTwo.focus();
	}
};

xTwo.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

xTwo.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		radiusTwo.focus();
	}
};

radiusTwo.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

radiusTwo.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
        xOne.focus();

	}
};

results.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
	}
};

scaleMenu.onchange = function () {
    "use strict";
    cScale = Number(scaleMenu.value);
    process();
};

xOne.value = "-1.0";
radiusOne.value = "1.0";
xTwo.value = "0.995";
radiusTwo.value = "1.0";
process();

xOne.focus();
