/*
	Planimeter III - A Mathematical Instrument
	Copyright © 2007-2019 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA	02110-1301	USA

	Planimeter III - browser version 1.2
	22 November, 2019
	Copyright © 2007-2019 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint browser, bitwise, for */

/*property
    altKey, areaFactor, backgroundColor, base, checked, clientX, clientY, color,
    innerHTML, length, onchange, onclick, ondblclick, onmousedown, onmousemove,
    onmouseout, onmouseup, opacity, open, ppiH, ppiV, preventDefault, push,
    round, scrollX, scrollY, setAttribute, shiftKey, sqrt, stopPropagation,
    style, substring, title, toFixed, toString, value
*/

import {getPreference, setPreference} from "./preferences.js";
import {newImage, newInput, newText, moveObj} from "./webWidget.js";
import {ruler} from "./pixelscale.js";

//////////////////////////////////////// GUI setup ///////////////////////////////////////

const planimeterTitle = (
	"Double-click to reset the counter.\n\n" +
    "To measure the area of a closed curve, place the tracer cross onto the curve, " +
    "reset the counter, then drag the cross once around the curve.\n\n" +
    "To measure the length of a path, instead of an area, hold down the alt-key " +
    "while resetting the counter. "
);

const polygonTitle = (
	"To measure the area of a polygon, double-click the counter to reset it.\n" +
	"Drag the tracer cross onto each vertex in turn - the coordinates are recorded on mouseup.\n\n" +
    "To measure the length of a path, instead of an area, hold down the alt-key " +
    "while resetting the counter. "
);

const style = "font-family:'Lucida Grande','Lucida Sans Unicode';color:white;font-weight:normal;background-color:black;border:1px solid black;text-align:center;font-size:16px;";

let areaText = newText(450, 90, 120, 20, "000000", 1, style);

let target = newImage(600, 80, 40, 40, "Resources/Images/target.png", 10, 1);
let dragSpot = newImage(600, 80, 40, 40, "Resources/Images/circle.png", 10, 0.13);

let modeKey = newInput(535, 1, 8, 16, "planimeter", 2);
modeKey.setAttribute("type", "checkbox");
modeKey.checked = (getPreference("planimeterPref") === "1");
modeKey.title = (
	modeKey.checked
	? "Choose polygon mode."
	: "choose planimeter mode"
);

let modeButtonText = (
	modeKey.checked
	? "planimeter"
	: "polygon"
);
let modeButton = newInput(450, 1, 8, 16, modeButtonText, 2);
modeButton.setAttribute("type", "button");
modeButton.title = (
	modeKey.checked
	? "Choose polygon mode."
	: "choose planimeter mode"
);

areaText.base = (
	modeKey.checked
	? planimeterTitle
	: polygonTitle
);
areaText.title = areaText.base;

function changeMode() {
	modeButton.value = (
		modeKey.checked
		? "planimeter"
		: "polygon"
	);
	modeButton.title = "Choose " + (
		modeKey.checked
		? "polygon"
		: "planimeter"
	) + " mode.";
	modeKey.title = "Choose " + (
		modeKey.checked
		? "polygon"
		: "planimeter"
	) + " mode.";

	areaText.title = (
		modeKey.checked
		? planimeterTitle
		: polygonTitle
	);

	setPreference("planimeterPref", (
		modeKey.checked
		? "1"
		: "0"
	));
}

modeKey.onchange = function () {
	changeMode();
};
modeButton.onclick = function () {
	modeKey.checked = !modeKey.checked;
	changeMode();
};

let helpButton = newImage(560, 0, 30, 22, "Resources/Images/help.png", 1, 1);
helpButton.title = "Displays help information about the planimeter and polygon modes.";

helpButton.onmousedown = function () {
	helpButton.style.opacity = "0.5";
};
helpButton.onmouseup = function () {
	helpButton.style.opacity = "1.0";
	window.open("Help2.html");
};

let clockwiseKey = newInput(550, 115, 8, 16, "anticlockwise", 1);
clockwiseKey.setAttribute("type", "checkbox");
clockwiseKey.checked = (getPreference("clockwisePref") === "1");

let clockwiseButton = newInput(455, 115, 8, 16, "anticlockwise", 1);
clockwiseButton.setAttribute("type", "button");

let clockwise;

function changeClockwise() {
	clockwiseButton.value = (
		clockwiseKey.checked
		? "clockwise"
		: "anticlockwise"
	);
	clockwiseButton.title = "Choose " + (
		clockwiseKey.checked
		? "anticlockwise"
		: "clockwise"
	) + " mode.";
	clockwiseKey.title = "Choose " + (
		clockwiseKey.checked
		? "anticlockwise"
		: "clockwise"
	) + " mode.";

	setPreference("clockwisePref", (
		clockwiseKey.checked
		? "1"
		: "0"
	));

	clockwise = (
		clockwiseKey.checked
		? 1
		: -1
	);
}
changeClockwise();

clockwiseKey.onchange = function () {
	changeClockwise();
};
clockwiseButton.onclick = function () {
	clockwiseKey.checked = !clockwiseKey.checked;
	changeClockwise();
};

////////////////////////////////////// Initial Setup /////////////////////////////////////

let deltaX = -4; 		// adjustments for the mouse pointer
let deltaY = -4;

let area = 0;

let oldX = 0;
let oldY = 0;

let areaScaleFactor = 1;

let downX = null;
let downY = null;

let distanceMode = false;

let dragEnabled = false;

let xCoord = [];
let yCoord = [];

function setColors(distanceMode) {
    function invColor(color) {
        let tmp = (parseInt(color.substring(1), 16) ^ 0xFFFFFF);
        let inv = tmp.toString(16);

        while (inv.length < 6) {
            inv = "0" + inv;
        }
        return "#" + inv;
    }

    if (distanceMode) {
        areaText.style.color = invColor(getPreference("textColorPref"));
        areaText.style.backgroundColor = invColor(getPreference("bkgdColorPref"));
    } else {
        areaText.style.color = getPreference("textColorPref");
        areaText.style.backgroundColor = getPreference("bkgdColorPref");
    }
}

function updateArea() {
    let val;

    if (ruler.areaFactor !== 0) {
    	areaScaleFactor = ruler.areaFactor / (ruler.ppiH * ruler.ppiV);
    } else {
    	areaScaleFactor = 1;
    }

    val = clockwise * area * areaScaleFactor;
    if (ruler.areaFactor !== 0) {
    	areaText.innerHTML = val.toFixed(3);
    } else {
    	areaText.innerHTML = Math.round(val);
    }
}

function pathLength(x, y) {
	let sum = 0;
	let dx;
	let dy;
	let i;
	let last = x.length - 1;

	for (i = 1; i <= last; i += 1) {
		dx = x[i] - x[i - 1];
		dy = y[i] - y[i - 1];
		sum += Math.sqrt(dx * dx + dy * dy);
	}

	dx = x[last] - x[0];
	dy = y[last] - y[0];
	sum += Math.sqrt(dx * dx + dy * dy);

	return sum;
}

function polyArea(x, y) {
	let sum = 0;
	let i;
	let last = x.length - 1;

	for (i = 1; i <= last; i += 1) {
		sum += 0.5 * (x[i - 1] * y[i] - x[i] * y[i - 1]);
	}
	sum += 0.5 * (x[last] * y[0] - x[0] * y[last]);

	return sum;
}

////////////////////////////////////// Event Handlers ////////////////////////////////////

areaText.ondblclick = function (event) {
    area = 0;
    areaText.innerHTML = "000000";
	distanceMode = event.altKey;
    setColors(distanceMode);

    xCoord = [];	// polygon mode
    yCoord = [];	// polygon mode
};

dragSpot.onmousedown = function (event) {
	event.stopPropagation();
    event.preventDefault();

    oldX = event.clientX + window.scrollX + deltaX;
    oldY = event.clientY + window.scrollY + deltaY;

    if (ruler.areaFactor !== 0) {
    	areaScaleFactor = ruler.areaFactor / (ruler.ppiH * ruler.ppiV);
    } else {
    	areaScaleFactor = 1;
    }

    dragEnabled = true;
};

dragSpot.onmousemove = function (event) {
    let mvX = event.clientX + window.scrollX + deltaX;
    let mvY = event.clientY + window.scrollY + deltaY;
    let dx;
    let dy;
    let val;

	event.stopPropagation();
    event.preventDefault();

    if (dragEnabled && modeKey.checked) {	// planimeter mode
    	if (event.altKey) {
    		if (downX === null) {
    			downX = mvX;
    		}
    		downY = null;
			moveObj(dragSpot, downX - 40, mvY - 40);
			moveObj(target, downX - 40, mvY - 40);
    	} else if (event.shiftKey) {
    		if (downY === null) {
    			downY = mvY;
    		}
    		downX = null;
 			moveObj(dragSpot, mvX - 40, downY - 40);
			moveObj(target, mvX - 40, downY - 40);
    	} else {
			moveObj(dragSpot, mvX - 40, mvY - 40);
			moveObj(target, mvX - 40, mvY - 40);
		}
    	dx = mvX - oldX;
    	dy = mvY - oldY;

    	if (distanceMode) {
        	area += Math.sqrt(dx * dx + dy * dy);
        	areaText.innerHTML = Math.round(area);
    	} else {
        	area += 0.25 * ((mvX + oldX) * dy - (mvY + oldY) * dx); // area += 0.5 * (mvX * dy - mvY * dx);
        	val = clockwise * area * areaScaleFactor;
    		if (ruler.areaFactor !== 0) {
    			areaText.innerHTML = val.toFixed(3);
			} else {
				areaText.innerHTML = Math.round(val);
			}
    	}

    	oldX = mvX;
    	oldY = mvY;
    }

    if (dragEnabled && !modeKey.checked) {	// polygon mode
 		moveObj(dragSpot, mvX - 40, mvY - 40);
		moveObj(target, mvX - 40, mvY - 40);

    	oldX = mvX;
    	oldY = mvY;
    }
};

dragSpot.onmouseup = function (event) {
	event.stopPropagation();
    event.preventDefault();

    if (modeKey.checked) {
    	downX = null;
    	downY = null;
    } else {
    	xCoord.push(oldX);
    	yCoord.push(oldY);

    	if (distanceMode) {
        	area = pathLength(xCoord, yCoord);
        	areaText.innerHTML = Math.round(area);
    	} else {
        	area = polyArea(xCoord, yCoord);
        	areaText.innerHTML = Math.round(clockwise * area);
    	}
    }

    dragEnabled = false;
};

dragSpot.onmouseout = function (event) {
	event.stopPropagation();
    event.preventDefault();

	downX = null;	// planimeter mode
	downY = null;	// planimeter mode

    dragEnabled = false;
};

////////////////////////////////////////// Exports ///////////////////////////////////////

export {areaText, updateArea};
