/*
    Pixel Scale - A Screen Ruler
    Copyright © 2007, 2017, 2019 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Pixel Scale - browser version 1.2
    22 November 219
    Copyright © 2007, 2017, 2019 Harry Whitfield
    mailto:g6auc@arrl.net
*/

/*jslint browser, devel */

/*property
    PI, altKey, areaFactor, backgroundColor, clientX, clientY, clip, cos,
    getElementById, height, innerHTML, isNaN, keyCode, maxW, minW, ondblclick,
    onkeypress, onmousedown, onmousemove, onmouseout, onmouseup, opacity, open,
    pixelMode, ppiH, ppiV, preventDefault, readonly, rotate, rotation, round,
    scaleFactor, scrollX, scrollY, setAttribute, shiftKey, sin, sqrt, src,
    srcWidth, stopPropagation, style, title, toFixed, value, which, width
*/

import {getPreference, setPreference} from "./preferences.js";
import {newImage, newInput, newTextArea, moveObj} from "./webWidget.js";
import {setMode} from "./scales.js";
import {updateArea} from "./planimeter.js";

//////////////////////////////////////// GUI Setup ///////////////////////////////////////

const xOrg = 900;	// initial position of the ruler
const yOrg = 80;

const rulerSrcWidth = 1500;
const rulerSrcHeight = 72;
const legendSrcHeight = 40;

const style = "font-family:Helvetica;color:white;font-weight:normal;background-color:black;border:none;text-align: center;font-size:16px;";

const rulerLength = Number(getPreference("rulerLengthPref"));

let ruler = newImage(xOrg, yOrg, rulerSrcWidth, rulerSrcHeight, "Resources/Images/rulr100_50.png", 21, "1.0", 0, 0);
ruler.setAttribute("style", "position: absolute;");
ruler.style.clip = "rect(0px," + rulerLength + "px,36px,0px)";
ruler.style.backgroundColor = getPreference("colorPref");
ruler.style.opacity = "1.0";
ruler.title = "Double-click to change scales.";

let legend = newImage(xOrg, yOrg + 16, rulerSrcWidth, legendSrcHeight, "Resources/Images/legd100.png", 22, "1.0", 0, -16);
legend.setAttribute("style", "position: absolute;");
legend.style.clip = "rect(0px," + rulerLength + "px,20px,0px)";
legend.style.backgroundColor = getPreference("legendColorPref");
legend.style.opacity = "1.0";
legend.title = "Double-click to change scales.";

let label = newImage(xOrg, yOrg + 16, 60, 20, "Resources/Images/P100_50.png", 23, 1, "1.0", -16);
label.title = "Double-click to change scales.";

let button = newImage(xOrg - 170, yOrg - 60, 240, 160, "Resources/Images/circle.png", 24, "1.0", 120, 80);
button.title = (
	"Drag to rotate ruler. Click with shift-key to set at 0, 90, 180, 270 degrees." +
	" Drag with alt-key to change ruler length. Drag with shift+alt-keys to calibrate.\n\n" +
	"Double-click then drag to move the ruler over the window. Double-click again to cancel."
);

let buttonText = newTextArea(xOrg - 70, yOrg + 7, 30, 20, "0", 25, style);
buttonText.readonly = "readonly";

let screenWidth = newInput(xOrg, yOrg - 55, 100, 20, Number(getPreference("screenHPref")).toFixed(3), 1, style);
screenWidth.title = "Enter the screen width (in inches), then press RETURN.";

let screenHeight = newInput(xOrg + 194, yOrg - 55, 100, 20, Number(getPreference("screenVPref")).toFixed(3), 1, style);
screenHeight.title = "Enter the screen height (in inches), then press RETURN.";

let screenPpiHValue = screen.width / Number(getPreference("screenHPref"));
let screenPpiH = newInput(xOrg, yOrg - 30, 100, 20, screenPpiHValue.toFixed(3), 1, style);
screenPpiH.title = "Enter the horizontal screen pixels per inch, then press RETURN.";

let screenPpiVValue = screen.height / Number(getPreference("screenVPref"));
let screenPpiV = newInput(xOrg + 194, yOrg - 30, 100, 20, screenPpiVValue.toFixed(3), 1, style);
screenPpiV.title = "Enter the vertical screen pixels per inch, then press RETURN.";

let helpButton = newImage(1260, 0, 30, 22, "Resources/Images/help.png", 1, 1);
helpButton.title = "Displays help information about the pixel scale widget.";

helpButton.onmousedown = function () {
	helpButton.style.opacity = "0.5";
};
helpButton.onmouseup = function () {
	helpButton.style.opacity = "1.0";
	window.open("Help3.html");
};

let dragIndicator = newImage(680, 92, 14, 14, "Resources/Images/red.png", 1, 1);
dragIndicator.title = "When green, indicates that the ruler can be dragged over the screen.";

let diags = document.getElementById("diags");

////////////////////////////////////// Initial Setup /////////////////////////////////////

function getClockwise() {
	return 2 * Number(getPreference("clockwisePref")) - 1;
}

function rotateAll(angle) {
	ruler.rotate(angle);
	legend.rotate(angle);
	label.rotate(angle);
}

function thePPI(theta, h, v, pixelMode) {	// theta in degrees
	let a;
	let b;

	if (pixelMode !== 0) {
    	return pixelMode;
    }

    a = h * Math.sin(theta * Math.PI / 180);
    b = v * Math.cos(theta * Math.PI / 180);
    return h * v / Math.sqrt(a * a + b * b);
}

function setWidth(width) {
    if (width < ruler.minW) {
    	width = ruler.minW;
    }
    if (width > ruler.maxW) {
    	width = ruler.maxW;
    }
//  width = ruler.minW * Math.ceil(width / ruler.minW) * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
	width = width * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
    ruler.style.clip = "rect(0px," + width + "px,36px,0px)";
   	legend.style.clip = "rect(0px," + width + "px,20px,0px)";
	setPreference("rulerLengthPref", String(width));
}

function calibrationDone() {
	let ppiH = ruler.ppiH;
	let ppiV = ruler.ppiV;

	screenH = screen.width / ppiH;
	screenV = screen.height / ppiV;

	setPreference("screenHPref", String(screenH));
	setPreference("screenVPref", String(screenV));

	screenWidth.value = screenH.toFixed(3);
	screenHeight.value = screenV.toFixed(3);

	screenPpiH.value = ppiH.toFixed(3);
	screenPpiV.value = ppiV.toFixed(3);
}

function normalized(rotation) {
    rotation = ((rotation % 360) + 360) % 360;
    if (rotation > 180) {
    	rotation = rotation - 360;
    }
    return rotation;
}

let clockwise = getClockwise();

let screenH = Number(getPreference("screenHPref"));	// in inches
let screenV = Number(getPreference("screenVPref"));	// in inches

ruler.maxW = rulerSrcWidth;
ruler.minW = 100;
ruler.pixelMode = 100;
ruler.scaleFactor = 100;
ruler.areaFactor = 0;
ruler.srcWidth = rulerSrcWidth;

ruler.ppiH = screen.width / screenH;
ruler.ppiV = screen.height / screenV;

ruler.rotation = Number(getPreference("rotationPref"));
rotateAll(ruler.rotation);

buttonText.value = ruler.rotation;

let resizing = false;
let calibrating = false;

let downX = 0;

let downCalibrationH = 0;
let downCalibrationV = 0;
let downRotation = 0;
let downWidth = 0;

const gCalibrationGain = 1.0;
const gRotationGain = 1.0;
const gWidthGain = 4.0;

setMode(getPreference("modePref"));

ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
legend.width = ruler.width;

////////////////////////////////////// Event Handlers ////////////////////////////////////

screenWidth.onkeypress = function (event) {
 	let x = event.which || event.keyCode;
 	let val;
 	let ppiH;

	if (x === 13) {
		val = parseFloat(screenWidth.value);
		if (!Number.isNaN(val) && (val >= 5) && (val <= 40)) {
			setPreference("screenHPref", String(val));
			screenH = val;
			ppiH = screen.width / screenH;
			screenPpiH.value = ppiH.toFixed(3);
			ruler.ppiH = ppiH;
			ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
			legend.width = ruler.width;
		} else {
			alert("Please enter a number between 4 and 40.");
			screenWidth.value = getPreference("screenHPref");
		}
	}
};

screenHeight.onkeypress = function (event) {
 	let x = event.which || event.keyCode;
 	let val;
 	let ppiV;

	if (x === 13) {
		val = parseFloat(screenHeight.value);
		if (!Number.isNaN(val) && (val >= 3) && (val <= 20)) {
			setPreference("screenVPref", String(val));
			screenV = val;
			ppiV = screen.height / screenV;
			screenPpiV.value = ppiV.toFixed(3);
			ruler.ppiV = ppiV;
			ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
			legend.width = ruler.width;
		} else {
			alert("Please enter a number between 3 and 30.");
			screenHeight.value = getPreference("screenVPref");
		}
	}
};

screenPpiH.onkeypress = function (event) {
 	let x = event.which || event.keyCode;
 	let val;
 	let ppiH;

	if (x === 13) {
		val = parseFloat(screenPpiH.value);
		if (!Number.isNaN(val) && (val >= 32) && (val <= 512)) {
			ppiH = val;
			screenH = screen.width / ppiH;
			setPreference("screenHPref", String(screenH));
			screenWidth.value = screenH.toFixed(3);
			ruler.ppiH = ppiH;
			ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
			legend.width = ruler.width;
		} else {
			alert("Please enter a number between 32 and 512.");
			val = screen.width / Number(getPreference("screenHPref"));
			screenPpiH.value = val.toFixed(3);
		}
	}
};

screenPpiV.onkeypress = function (event) {
 	let x = event.which || event.keyCode;
 	let val;
 	let ppiV;

	if (x === 13) {
		val = parseFloat(screenPpiV.value);
		if (!Number.isNaN(val) && (val >= 32) && (val <= 512)) {
			ppiV = val;
			screenV = screen.height / ppiV;
			setPreference("screenVPref", String(screenV));
			screenHeight.value = screenV.toFixed(3);
			ruler.ppiV = ppiV;
			ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
			legend.width = ruler.width;
		} else {
			alert("Please enter a number between 32 and 512.");
			val = screen.height / Number(getPreference("screenVPref"));
			screenPpiV.value = val.toFixed(3);
		}
	}
};


let buttonEnabled = false;
let dragEnabled = false;

button.ondblclick = function () {
	dragEnabled = !dragEnabled;
	dragIndicator.src = (
		dragEnabled
		? "Resources/Images/green.png"
		: "Resources/Images/red.png"
	);
};

button.onmousedown = function (event) {
    event.stopPropagation();
    event.preventDefault();

    downX = event.clientX + window.scrollX;
    downRotation = ruler.rotation;
    downWidth = Number(getPreference("rulerLengthPref"));
    downCalibrationH = Number(getPreference("hAdjustPref"));
    downCalibrationV = Number(getPreference("vAdjustPref"));

    clockwise = getClockwise();

 	if (event.altKey && event.shiftKey) {
        ruler.rotation = ((ruler.rotation % 360) + 360) % 360;
        if (((ruler.rotation % 90) === 0) && (ruler.pixelMode === 0)) {
        	calibrating = true;
        } else {
        	alert("Calibration can only be done when the ruler is horizontal or vertical and is in a centimetre or inch mode.");
        }
        buttonEnabled = true;
    } else if (event.shiftKey) {
        if (ruler.rotation % 90 === 0) {
        	ruler.rotation = (ruler.rotation + 90) % 360;
        	legend.rotation = ruler.rotation;
    		label.rotation = ruler.rotation;
    	} else {
    		ruler.rotation = 0;
    		legend.rotation = 0;
    		label.rotation = 0;
    	}
        rotateAll(ruler.rotation);
       	buttonText.value = normalized(clockwise * ruler.rotation);
        setPreference("rotationPref", buttonText.value);
    } else if (event.altKey) {
        ruler.rotation = 0;
        legend.rotation = 0;
        label.rotation = 0;
        rotateAll(ruler.rotation);
		resizing = true;
        buttonText.value = 0;
        setPreference("rotationPref", buttonText.value);
        buttonEnabled = true;
    } else {
        buttonEnabled = true;
    }
};

button.onmouseup = function () {
    event.stopPropagation();
    event.preventDefault();

	buttonEnabled = false;

    if (resizing) {
    	ruler.rotation = downRotation;
    	legend.rotation = downRotation;
    	label.rotation = downRotation;
    	rotateAll(ruler.rotation);
    	resizing = false;
    	buttonText.value = normalized(clockwise * downRotation);
    } else if (calibrating) {
    	calibrationDone();
    	calibrating = false;
    }
};
button.onmouseout = button.onmouseup;


function reSize(wDeltaX) {
    let gain = gWidthGain;
    let width;

    width = Math.round(downWidth + gain * wDeltaX);
    setWidth(width);
}

function calibrate(cDeltaX) {
    let gain = gCalibrationGain;
    let hAdjustPref;
    let vAdjustPref;

    if ((ruler.rotation % 180) === 0) {   // horizontal calibration
        hAdjustPref = downCalibrationH + gain * cDeltaX;
        if (hAdjustPref < -100) {
        	hAdjustPref = -100;
        } else if (hAdjustPref > 100) {
        	hAdjustPref = 100;
        }
        setPreference("hAdjustPref", String(hAdjustPref));
        ruler.ppiH = (1 + 0.0001 * hAdjustPref) * screen.width / screenH;
        diags.innerHTML = "hAdjustPref: " + String(hAdjustPref) + ", " + "ppiH: " + ruler.ppiH.toFixed(3);
	} else {   // vertical calibration
        vAdjustPref = downCalibrationV + gain * cDeltaX;
        if (vAdjustPref < -100) {
        	vAdjustPref = -100;
        } else if (vAdjustPref > 100) {
        	vAdjustPref = 100;
        }
        setPreference("vAdjustPref", String(vAdjustPref));
        ruler.ppiV = (1 + 0.0001 * vAdjustPref) * screen.height / screenV;
        diags.innerHTML = "vAdjustPref: " + String(vAdjustPref) + ", " + "ppiV: " + ruler.ppiV.toFixed(3);
    }
    ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
    legend.width = ruler.width;

	updateArea();
}

function rotate(rDeltaX) {
    let gain = gRotationGain;

    ruler.rotation = Math.round(360 + downRotation + gain * rDeltaX) % 360;
    label.rotation = ruler.rotation;
    legend.rotation = ruler.rotation;
    rotateAll(ruler.rotation);

	ruler.width = ruler.srcWidth * thePPI(ruler.rotation, ruler.ppiH, ruler.ppiV, ruler.pixelMode) / ruler.scaleFactor;
	legend.width = ruler.width;

    buttonText.value = normalized(clockwise * ruler.rotation);
    setPreference("rotationPref", buttonText.value);
}

function relocate(mvX, mvY) {
	mvX += 90;
	mvY -= 35;

	moveObj(ruler, mvX, mvY);
	moveObj(legend, mvX, mvY + 16);
	moveObj(label, mvX, mvY + 16);
	moveObj(button, mvX - 170, mvY - 60);

	moveObj(buttonText, mvX - 70, mvY + 7);
}

button.onmousemove = function (event) {
	let mvX = event.clientX + window.scrollX;
	let mvY = event.clientY + window.scrollY;

    event.stopPropagation();
    event.preventDefault();

   	if (buttonEnabled) {
    	if (resizing) {
    		reSize(mvX - downX);
    	} else if (calibrating) {
    		calibrate(mvX - downX);
    	} else if (dragEnabled) {
			relocate(mvX, mvY);
    	} else {
    		rotate(mvX - downX);
    	}
    }
};

function noAction(event) {
    event.stopPropagation();
    event.preventDefault();
}

ruler.onmousedown = noAction;
legend.onmousedown = noAction;
ruler.onmouseup = noAction;
legend.onmouseup = noAction;
ruler.onmouseout = noAction;
legend.onmouseout = noAction;
ruler.onmousemove = noAction;
legend.onmousemove = noAction;

export {ruler, legend, label, thePPI};
