// Convert Unix Time - version 1.6.1
// 2 May, 2015
// Copyright 2004-2015 Harry Whitfield
// mailto:g6auc@arrl.net

/* global report, reportHOffset */

/*properties
    floor, getTime, getTimezoneOffset, length, round, toFixed, toLocaleString,
    toString, toUTCString
*/

function TZ(secs) {
	var neg = (secs < 0),
		hrs,
		min;
	
	if (neg) { secs = -secs; }
	hrs = Math.floor(secs / 3600).toString();
	min = Math.round((secs % 3600) / 60).toString();
	if (hrs.length === 1) { hrs = "0" + hrs; }
	if (min.length === 1) { min = "0" + min; }
	if (neg) { return "-" + hrs + min; }
	return "+" + hrs + min;
}

function UnixTime() {			// seconds from 0000 UTC on 1 Jan 1970
	var d = new Date();
	
	return Math.floor(d.getTime() / 1000);
}

function LocalTime() {
	var d = new Date();
	
	return d.toString();
}
			
function convertToLocale(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return d.toLocaleString() + '  ' + TZ(-60 * d.getTimezoneOffset());
}

function convertToLocal(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return d.toString();
}

function UniversalTime() {
	var d = new Date();
	
	return d.toUTCString();
}

function convertToUTC(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return d.toUTCString();
}

function NTPTime() {			// seconds from 0000 UTC on 1 Jan 1900
	var d = new Date();			// NTPTime = UnixTime + 2208988800

	return 2208988800 + Math.floor(d.getTime() / 1000);
}

function convertToNTPTime(UnixTimeStamp) {
	return (2208988800 + Number(UnixTimeStamp)).toFixed(0);
}

function MacintoshTime() {		// seconds from 0000 on 1 Jan 1904
	var d = new Date();			// MacintoshTime = UnixTime + 2082844800 + TimeZone
	
	return 2082844800 + Math.floor(d.getTime() / 1000) - 60 * d.getTimezoneOffset();
}

function convertToMacintoshTime(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return (2082844800 + Math.floor(d.getTime() / 1000) - 60 * d.getTimezoneOffset()).toFixed(0);
}

function TimeZone() {			// seconds East of Greenwich
	var d = new Date();			// localTime = UTC + TimeZone

	return -60 * d.getTimezoneOffset();
}

function convertTimeZone(UnixTimeStamp) {		// seconds East of Greenwich
	var d = new Date(1000 * UnixTimeStamp);		// localTime = UTC + TimeZone
	
	return -60 * d.getTimezoneOffset();
}

function InternetTime() {
	var d = new Date(),
		time = Math.floor((d.getTime() + 3600000) % 86400000 / 864),
		cents = time % 100,
		beats = ((time - cents) / 100) % 1000,
		res1 = beats.toString(),
		res2;
		
	if (res1.length === 1) { res1 = "00" + res1; } else if (res1.length === 2) { res1 = "0" + res1; }
	res2 = cents.toString();
	if (res2.length === 1) { res2 = "0" + res2; }
	return "@" + res1 + "." + res2;
}

function convertToInternetTime(UnixTimeStamp) {
	var time = Math.floor((1000 * UnixTimeStamp + 3600000) % 86400000 / 864),
		cents = time % 100,
		beats = ((time - cents) / 100) % 1000,
		res1 = beats.toString(),
		res2;
		
	if (res1.length === 1) { res1 = "00" + res1; } else if (res1.length === 2) { res1 = "0" + res1; }
	res2 = cents.toString();
	if (res2.length === 1) { res2 = "0" + res2; }
	return "@" + res1 + "." + res2;
}

function julianDate() {			// days from noon UTC on 1 Jan 4713 BCE
	var d = new Date();
	
	return (2415020.5 + ((2208988800 + Math.floor(d.getTime() / 1000)) / 86400)).toFixed(6);
}
    
function convertToJulianDate(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return (2415020.5 + ((2208988800 + Math.floor(d.getTime() / 1000)) / 86400)).toFixed(6);
}

function ModifiedJulianDate() {	//  MJD = JD - 2400000.5
	var d = new Date();
	
	return (15020 + ((2208988800 + Math.floor(d.getTime() / 1000)) / 86400)).toFixed(6);
}
    
function convertToModifiedJulianDate(UnixTimeStamp) {
	var d = new Date(1000 * UnixTimeStamp);
	
	return (15020 + ((2208988800 + Math.floor(d.getTime() / 1000)) / 86400)).toFixed(6);
}

function GPSweek() {
	var w = ((julianDate() - 2444244.5) / 7);
	
	return (w - 1024 * Math.floor(w / 1024)).toFixed(6);
}

function convertToGPSweek(UnixTimeStamp) {
	var w = ((convertToJulianDate(UnixTimeStamp) - 2444244.5) / 7);
	return (w - 1024 * Math.floor(w / 1024)).toFixed(6);
}

function convertTime(UnixTimeStamp, mode) {
	switch (mode) {
	case 'Unix Time':
		return UnixTimeStamp;
	case 'Universal Time':
		return convertToUTC(UnixTimeStamp);
	case 'Local Time':
		return convertToLocale(UnixTimeStamp);
	case 'NTP Time':
		return convertToNTPTime(UnixTimeStamp);
	case 'Macintosh Time':
		return convertToMacintoshTime(UnixTimeStamp);
	case 'Internet Time':
		return convertToInternetTime(UnixTimeStamp);
	case 'Julian Date':
		return convertToJulianDate(UnixTimeStamp);
	case 'Modified Julian Date':
		return convertToModifiedJulianDate(UnixTimeStamp);
	case 'GPS Week':
		return convertToGPSweek(UnixTimeStamp);
	default:
		return 'Invalid Mode.';
	}
}

/*
function setTime(data, mode) {
  	var input = parseInt(data, 10);
  	
   	if (isNaN(input)) {
   		beep();
   		report.data = 'Invalid Input';
   	} else {
  		report.data = mode;
  		report.hOffset = reportHOffset();
  		command.data = input + '  =>  ' + convertTime(input, mode);
  	}
}
*/
