/*
	ConvertTime - Date and Time Converter
	Copyright © 2004-2015 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
	
	ConvertTime - browser version 1.0
	2 May, 2015
	Copyright © 2004-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global paramOne, paramTwo, paramThree, paramFour, headingFour, Menu, convertTime
*/

/*properties
    day, floor, getDate, getFullYear, getHours, getMinutes, getMonth, getSeconds,
    getTime, hours, keyCode, length, minutes, month, onchange, onkeypress,
    seconds, split, toString, toUTCString, value, which, year
*/

// map custom names onto model names
var localdate     = paramOne;
var localtime     = paramTwo;
var universaltime = paramThree;
var outputtime    = paramFour;
/*
var localdateHeading     = headingOne;
var localtimeHeading     = headingTwo;
var universaltimeHeading = headingThree;
*/
var outputtimeHeading    = headingFour;

var convModeMenu = Menu[0];
var updateMenu   = Menu[1];

var typePref = 'Unix Time';
var tickPref = 'Never';

var gDate;
var clockOn  = false;
var fastTick = false;
var intervalID = null;

/*
	['GPS Week', 'Internet Time', 'Julian Date', 'Local Time', 'Macintosh Time',
		'Modified Julian Date', 'NTP Time', 'Universal Time', 'Unix Time'],
	['Once a Second', 'Once a Minute', 'Never']
*/

function TheDate(year, month, day) {
	this.year  = year;
	this.month = month;
	this.day   = day;
}
	
function TheTime(hours, minutes, seconds) {
	this.hours   = hours;
	this.minutes = minutes;
	this.seconds = seconds;
}

function todayDate() {
	gDate  = new Date();
	return new TheDate(gDate.getFullYear(), gDate.getMonth() + 1, gDate.getDate());
}

function todayTime() {
	return new TheTime(gDate.getHours(), gDate.getMinutes(), gDate.getSeconds());
}

function leapYear(year) {
	if ((year %   4) !== 0) { return false; }
	if ((year % 400) === 0) { return true; }
	if ((year % 100) === 0) { return false; }
	return true;
}

function getDaysIn(month, year) {	// month 1..12
	var daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
	
	if (month !== 2) { return daysInMonth[month - 1]; }
	if (leapYear(year)) { return 29; }
	return 28;
}

function pad(x, n) {
	x = x.toString();
	while (x.length < n) { x = "0" + x; }
	return x;
}

function parseDate(s) {
	var invalidDate = new TheDate(-1, 0, 0),
		item  = s.split('/', 3),
		year  = parseInt(item[0], 10),
		month,
		day,
		maxd;

	if (isNaN(year)) { return invalidDate; }
	
	month = parseInt(item[1], 10);
	if (isNaN(month)) { return invalidDate; }
	day   = parseInt(item[2], 10);
	if (isNaN(day)) { return invalidDate; }
	if ((year  < 1970) || (year  > 9999)) { return invalidDate; }
	if ((month <    1) || (month >   12)) { return invalidDate; }

	maxd = getDaysIn(month, year);

	if ((day   <    1) || (day   > maxd)) { return invalidDate; }
	return new TheDate(year, month, day);
}

function parseTime(s) {
	var invalidTime = new TheTime(-1, 0, 0),
		item = s.split(':', 3),
		hours   = parseInt(item[0], 10),
		minutes,
		seconds;
		
	if (isNaN(hours)) { return invalidTime; }
	minutes = parseInt(item[1], 10);
	if (isNaN(minutes)) { return invalidTime; }
	seconds = parseInt(item[2], 10);
	if (isNaN(seconds)) { return invalidTime; }
	if ((hours   < 0) || (hours   > 23)) { return invalidTime; }
	if ((minutes < 0) || (minutes > 59)) { return invalidTime; }
	if ((seconds < 0) || (seconds > 59)) { return invalidTime; }
	
	return new TheTime(hours, minutes, seconds);
}

function displayTime() {	// clock mode
	var secs = 0,
		timerInterval, 
		d = new Date(),
		year,
		month,
		day,
		hours,
		minutes,
		seconds,
		ms = 0;
		
	if (!fastTick) {
		if (intervalID !== null) { clearInterval(intervalID); intervalID = null; }
		secs = d.getSeconds();
		ms = d.getMilliseconds();
		timerInterval = 60 - secs;
		if (timerInterval > 10) { timerInterval -= 5; }
		timerInterval = 1000 * timerInterval - ms;
		intervalID = setInterval(displayTime, timerInterval);
	}

	if (fastTick || (secs <= 1) || (secs > 58)) {
		year = String(d.getFullYear());
		month = String(d.getMonth() + 1);
		if (month.length === 1) { month = '0' + month; }
		day = String(d.getDate());
		if (day.length === 1) { day = '0' + day; }
		localdate.value = year + '/' + month + '/' + day;
	
		hours = String(d.getHours());
		if (hours.length === 1) { hours = '0' + hours; }
		minutes = String(d.getMinutes());
		if (minutes.length === 1) { minutes = '0' + minutes; }
		seconds = String(d.getSeconds());
		if (seconds.length === 1) { seconds = '0' + seconds; }
		localtime.value = hours + ':' + minutes + ':' + seconds;
	
   		universaltime.value = d.toUTCString();
		outputtime.value    = convertTime(Math.floor(d.getTime() / 1000), typePref);
	}
}

function universalDateTime(localDate, localTime) {
	var d = new Date(localDate.year, localDate.month - 1, localDate.day, localTime.hours, localTime.minutes, localTime.seconds),
		UnixTimeStamp = Math.floor(d.getTime() / 1000);
		
	return convertTime(UnixTimeStamp, 'Universal Time');
}

function convertedTime(localDate, localTime, typePref) {
	var d = new Date(localDate.year, localDate.month - 1, localDate.day, localTime.hours, localTime.minutes, localTime.seconds),
		UnixTimeStamp = Math.floor(d.getTime() / 1000);
		
	return convertTime(UnixTimeStamp, typePref);
}

function displayData(localDate, localTime) {
	if ((localDate.year < 0) || (localTime.hours < 0)) {
		universaltime.value = '';
		outputtime.value    = '';
		
		if (localDate.year  < 0) { universaltime.value = 'Invalid date'; }
		if (localTime.hours < 0) { outputtime.value    = 'Invalid time'; }
		return;
	}

	localdate.value = pad(localDate.year,  4) + '/' + pad(localDate.month,   2) + '/' + pad(localDate.day,     2);
   	localtime.value = pad(localTime.hours, 2) + ':' + pad(localTime.minutes, 2) + ':' + pad(localTime.seconds, 2);
   	 		
   	universaltime.value = universalDateTime(localDate, localTime);
	outputtime.value    = convertedTime(localDate, localTime, typePref);
}

function setToday() {
	displayData(todayDate(), todayTime());
}

function updatePrefs() {
	outputtimeHeading.value = typePref;
	clockOn  = (tickPref !== 'Never');
	fastTick = (tickPref === "Once a Second");
	setToday();
	
	if (intervalID !== null) { clearInterval(intervalID); intervalID = null; }

	if (clockOn) {	
		intervalID = setInterval(displayTime, 1000);
	}
}

convModeMenu.onchange  = function () {
	typePref = this.value;
	updatePrefs();
};

updateMenu.onchange  = function () {
	tickPref = this.value;
	updatePrefs();
};

localdate.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { 
		displayData(parseDate(localdate.value), parseTime(localtime.value));
	} else {
		universaltime.value = "";
		outputtime.value = "";
	}
};

localtime.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) {
		displayData(parseDate(localdate.value), parseTime(localtime.value));
	} else {
		universaltime.value = "";
		outputtime.value = "";
	}
};

updatePrefs();
