/*
	Sexagesimal Functions
 	Copyright © 2023 Harry Whitfield

 	This program is free software; you can redistribute it and/or modify it
 	under the terms of the GNU General Public License as published by the
 	Free Software Foundation; either version 2 of the License, or (at your
 	option) any later version.

 	This program is distributed in the hope that it will be useful, but
 	WITHOUT ANY WARRANTY; without even the implied warranty of
 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 	General Public License for more details.

 	You should have received a copy of the GNU General Public License along
 	with this program; if not, write to the Free Software Foundation, Inc.,
 	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

 	Base Converter - browser version 1.3.2
 	15 March, 2023
 	Copyright © 2023 Harry Whitfield
 	mailto:g6auc@arrl.net

	See Wikipedia at https://en.wikipedia.org/wiki/Sexagesimal
*/

/*property
    PI, floor, forEach, length, match, pow, push, reverse, split, sqrt,
    toString
*/

function parseSexagesimal(s) {
    var lookFor = /^((\d{1,2},)*\d{1,2})(;(\d{1,2}(,\d{1,2})*))?$/;
    var found = s.match(lookFor);
    var good = false;
    var value = 0;
    var mult = 1;

    if (found !== null) {
		good = true;

     	found[1].split(",").forEach(function (ele) {
    		ele = Number(ele);
    		if (ele > 59) {
    			good = false;
    		} else {
    			value = 60 * value + ele;
    		}
    	});

    	if (found[4] !== undefined) {
    		found[4].split(",").forEach(function (ele) {
     			ele = Number(ele);
   				if (ele > 59) {
    				good = false;
    			} else {
    				mult = mult / 60;
    				value += mult * ele;
    			}
    		});
    	}
    }
    if (good) {
    	return value;
    } else {
    	return NaN;
    }
}

function expand(val, base) {	// convert value to array of digit values
	var arr = [];

	if (val === 0) {
		return [0];
	}

	while (val > 0) {
		arr.push(val % base);
		val = (val - val % base) / base;
	}
	return arr.reverse();
}

function toSexaString(x, digits) {	// digits 0..8
	var whole = Math.floor(x);
	var fraction = x - whole;
	var count = digits || 0;
	var div = 60;
	var s = "";
	var digit;
	var rounding;

	if (count > 8) {
		count = 8;
	}

	expand(whole, 60).forEach(function (ele, i, arr) {
		s += ele.toString();
		if (i < arr.length - 1) {
			s += ",";
		}
	});

	if ((fraction > 0) && (count > 0)) {
		s += ";";
		rounding = 30 / Math.pow(60, count + 1);
		fraction += rounding;
		while (count > 0) {
			digit = Math.floor(div * fraction);
			fraction -= digit / div;
			s += String(digit);
			div = div * 60;
			count -= 1;
			if (count > 0) {
				s += ",";
			}
		}
	}
	return s;
}

/*
// Tests follow

var eprint;

eprint(parseSexagesimal("34"));										// 34
eprint(parseSexagesimal("34,59"));									// 2099
eprint(parseSexagesimal("34,59,42"));								// 125982
eprint(parseSexagesimal("34,59;42"));								// 2099.7
eprint(parseSexagesimal("34,59;42,6"));								// 2099.7016666666664

// expansion of √2 begins
eprint(parseSexagesimal("1;24,51,10,7,46,6,4,44"));					// 1.4142135623730903	cf 1.4142135623730951

// expansion of 2π begins
eprint(parseSexagesimal("6;16,59,28,1,34,51,46,14,49,55,12,35"));	// 6.283185307179586	cf 6.283185307179586


eprint(toSexaString(34, 0));					// 34
eprint(toSexaString(2099, 1));					// 34,59;0
eprint(toSexaString(125982, 3));				// 34,59,42;0,0,0
eprint(toSexaString(2099.7, 1));				// 34,59;42
eprint(toSexaString(2099.7016666666664, 2));	// 34,59;42,6

eprint(toSexaString(1.4142135623730903, 8));	// 1;24,51,10,7,46,6,4,44
eprint(toSexaString(6.283185307179586, 12));	// 6;16,59,28,1,34,51,46,15

eprint(toSexaString(Math.sqrt(2), 8));			// 1;24,51,10,7,46,6,4,44
eprint(toSexaString(2 * Math.PI, 12));			// 6;16,59,28,1,34,51,46,15
*/
