/*
	Base Converter
 	Copyright © 2023 Harry Whitfield

 	This program is free software; you can redistribute it and/or modify it
 	under the terms of the GNU General Public License as published by the
 	Free Software Foundation; either version 2 of the License, or (at your
 	option) any later version.

 	This program is distributed in the hope that it will be useful, but
 	WITHOUT ANY WARRANTY; without even the implied warranty of
 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 	General Public License for more details.

 	You should have received a copy of the GNU General Public License along
 	with this program; if not, write to the Free Software Foundation, Inc.,
 	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

 	Base Converter - browser version 1.3.2
 	15 March, 2023
 	Copyright © 2023 Harry Whitfield
 	mailto:g6auc@arrl.net
*/

/*jslint browser, this */

/*global paramOne, paramTwo, paramThree, paramFour,
	headingOne, headingTwo, headingThree, headingFour, clearData,
	execButton, eprint, lprint
*/

/*property
    every, floor, forEach, indexOf, isNaN, isSafeInteger, keyCode, onkeydown,
    onkeypress, onmousedown, onmouseup, opacity, preventDefault, push, reverse,
    round, select, split, stopPropagation, style, substring, toFixed,
    toUpperCase, value, which
*/

// map custom names onto model names
var numberBase = paramOne;
var numberRepr = paramTwo;
var newBase = paramThree;
var calculatedRepr = paramFour;

var numberBaseHeading = headingOne;
var numberReprHeading = headingTwo;
var newBaseHeading = headingThree;
var calculatedReprHeading = headingFour;

/*
Number.prototype.inRange = function (a, b) {
    return (this >= a) && (this <= b);
}
*/

/////////////////////////////////////////////////////////////////////////////////////////

var parseSexagesimal;
var toSexaString;

function expand(val, base) {	// convert value to array of digit values
	var arr = [];

	if (val === 0) {
		return [0];
	}

	while (val > 0) {
		arr.push(val % base);
		val = (val - val % base) / base;
	}
	return arr.reverse();
}

function xprint(val, base, digits) {	// print (natural) numbers in bases 2..36 or 60
	var c = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	var arr;
	var s = "";

	if (base === 60) {
		if (digits > 8) {
			digits = 8;
		}
		return toSexaString(val, digits);
	}

	if ((base === 10) && (Math.round(val) !== val)) {
		if (digits > 20) {
			digits = 20;
		}
		return val.toFixed(digits);
	}

	val = Math.floor(val);
	arr = expand(val, base);

	arr.forEach(function (ele) {
		s += c[ele];
	});

	return s;
}

function goodNumber(s, base) {
	var cc = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	var valid;
	var num;

	if (base === 60) {	// allow fractional part
		num = parseSexagesimal(s);
		return !Number.isNaN(num);	// && (s.indexOf(";") < 0);
	}

	valid = cc.substring(0, base);

	return s.toUpperCase().split("").every(function (c) {
		return valid.indexOf(c) !== -1;
	});
}

/////////////////////////////////////////////////////////////////////////////////////////

function process() {
    var base = parseInt(numberBase.value, 10);
    var oldNumber;
    var newbase;
    var decNumber = null;
    var digits = 20;
    var idx;

   	if (Number.isNaN(base)) {
        calculatedRepr.value = "Invalid Number Base";
        return;
    }
    if ((base < 2) || (base > 36)) {
        if (base !== 60) {
        	calculatedRepr.value = "Number Base Out of Range 2..36 or 60";
        	return;
        }
    }

    if (!goodNumber(numberRepr.value, base)) {
    	if (base === 10) {
    		decNumber = parseFloat(numberRepr.value);
    		if (Number.isNaN(decNumber)) {
    			calculatedRepr.value = "Bad Number Representation";
    			return;
    		}
    	} else {
			calculatedRepr.value = "Bad Number Representation";
        	return;
        }
    }

    if (base === 60) {
    	oldNumber = parseSexagesimal(numberRepr.value);
    } else if (decNumber !== null) {
    	oldNumber = decNumber;
    } else {
    	oldNumber = parseInt(numberRepr.value, base);
    }

    if (Number.isNaN(oldNumber)) {
        calculatedRepr.value = "Invalid Number Representation";
        return;
    }

    if (oldNumber < 0) {
        calculatedRepr.value = "Number must be non-negative.";
        return;
    }

	if (oldNumber > 9007199254740991) {	// 2^53 - 1
        calculatedRepr.value = "Number must be less than 2^53 ≈ 9x10^15.";
        return;
    }

	newbase = parseInt(newBase.value, 10);

    if (Number.isNaN(newbase)) {
        calculatedRepr.value = "Invalid New Base";
        return;
    }
    if ((newbase < 2) || (newbase > 36)) {
        if (newbase !== 60) {
        	calculatedRepr.value = "New Base Out of Range 2..36 or 60";
        	return;
        }
    }

    idx = newBase.value.indexOf(":");
    if (idx !== -1) {
    	digits = parseInt(newBase.value.substring(idx + 1), 10);
    	if (Number.isNaN(digits)) {
    		calculatedRepr.value = "Invalid New Base";
    	} else {
    		if (digits < 0) {
    			digits = 0;
    		}
    		if (digits > 20) {
    			digits = 20;
    		}
    	}
    }

    calculatedRepr.value = xprint(oldNumber, newbase, digits);

	eprint("Number Base: " + base);
	eprint("Number Representation: " + numberRepr.value);
	eprint("New Base [:Digits]: " + newBase.value);
	eprint("Converted Representation: " + calculatedRepr.value);
	eprint("---- ----- ----");
}

function test_case() {
	numberBase.value = 10;
	numberRepr.value = 144;
	newBase.value = 12;
	calculatedRepr.value = "";
	process();
}

var Return = 13;
var Tab = 9;
var Backspace = 8;

numberBase.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
        calculatedRepr.value = "";
    }
};

numberRepr.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
		calculatedRepr.value = "";
    }
};

newBase.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
        calculatedRepr.value = "";
    }
};

calculatedRepr.onkeydown = function (event) {
	var x = event.which || event.keyCode;

 	if (x === Return) {
		process();
	} else if (x === Tab) {
    	event.preventDefault();
		event.stopPropagation();
		numberBase.select();
    } else if (x !== Backspace) {
    	event.preventDefault();
		event.stopPropagation();
    }
};

execButton.onmousedown = function () {
	execButton.style.opacity = "0.5";
};
execButton.onmouseup = function () {
	execButton.style.opacity = "1.0";
	process();
};

lprint("Base Converter web widget started.");
eprint("---- ----- ----");
test_case();
